import math
from typing import Iterable


class InfoConv1D:
    """    
    arguments
    
     W        int            axis size
     KW       int            kernel size
     
     stride      int
     dilation    int
     
     padding     'valid'        no padding
                 'same'         output size will be the same 
                                or divided by stride
                 int            padding value for all sides
                 
                 Iterable of 4 ints 
                                paddings for left,right sides
 
    errors during the construction:
    
        ValueError
    
    result:
    
        .PADL .PADR  paddings for W axis

        .OW    result axis
        
        .OW_T  result transposed axes. 
                    it is None if padding != 'valid','same'
    """

    __slots__ = ['PADL', 'PADR', 'OW', 'OW_T']

    def __init__(self, W, KW, stride, dilation, padding):
        # Effective kernel size with dilation
        EKW = (KW-1)*dilation + 1        

        # Determine pad size of sides
        OW_T = None
        if padding == 'valid':
            PADL = PADR = 0
            OW_T =  W * stride + max(EKW - stride, 0)
        elif padding == 'same':
            PADL = int(math.floor((EKW - 1)/2))
            PADR = int(math.ceil((EKW - 1)/2))
            OW_T = W * stride
        elif isinstance(padding, int):
            PADL = PADR = padding
        elif isinstance(padding, Iterable):
            padding = tuple(int(x) for x in padding)
            if len(padding) != 2:
                raise ValueError("Invalid paddings list length.")
            PADL, PADR = padding
        else:
            raise ValueError("Invalid padding value.")

        self.PADL = PADL
        self.PADR = PADR
        
        self.OW = max(1, int((W + PADL + PADR - EKW) / stride + 1) )
        self.OW_T = OW_T
