from datetime import datetime
from pathlib import Path
from typing import Generator, Iterable, List, Sequence, Set, Union

import cv2
import numpy as np

from xlib import ax, mx, path as lib_path, cv as lib_cv

from RxTimeline import RxTimeline

from enum import Enum, auto



    

#TODO RxMediaSource
class RxMediaSource(mx.Disposable):
    class SourceType(Enum):
        Unselected = auto()
        VideoFile = auto()
        ImageSequence = auto()
        
    def __init__(self, state : dict = None):
        """
        """
        super().__init__()
        
        self._main_thread = ax.get_current_thread()
        self._sub_thread = ax.Thread('sub_thread')
        self._tg = ax.TaskGroup()
        
        self._mx_error = mx.TextState().dispose_with(self)
        self._mx_frame = mx.NumpyNoneableState(None).dispose_with(self)
        
        self._mx_tl : RxTimeline = None
        self._mx_media_pathdlg = None
        self._mx_source_type = mx.SingleChoice[RxMediaSource.SourceType](RxMediaSource.SourceType.Unselected, avail=RxMediaSource.SourceType, filter=self._mx_source_type_filter).dispose_with(self)
        
    def dispose(self) -> None:
        self._mx_source_type.set(self.SourceType.Unselected)
        super().dispose()
        self._tg.dispose()
        self._sub_thread.dispose()
        
    @property
    def mx_source_type(self) -> mx.SingleChoice[SourceType]:
        return self._mx_source_type    
        
    @property
    def mx_error(self) -> mx.TextState:   
        """Error set only when mx_source_type is selected, mx_media_path set to None and if something goes wrong"""
        return self._mx_error    
        
    @property
    def mx_media_pathdlg(self) -> Union[mx.PathState, None]: 
        """
        Avail when mx_source_type != SourceType.Unselected
        """   
        return self._mx_media_pathdlg
    
    @property
    def mx_timeline(self) -> Union[RxTimeline, None]:
        """Avail when _mx_media_pathdlg is opened"""
        return self._mx_tl
        
    @property
    def mx_frame(self) -> mx.NumpyNoneableState:
        """"""
        return self._mx_frame
      
    
        
    def _mx_source_type_filter(self, new_source_type : SourceType, source_type : SourceType) -> SourceType:
        print('_mx_source_type_filter', new_source_type)
        
        if source_type is not self.SourceType.Unselected:
            self._mx_media_pathdlg = self._mx_media_pathdlg.dispose()
        
        if new_source_type is self.SourceType.VideoFile:
            self._mx_media_pathdlg = mx.PathState( config=mx.PathState.Config(extensions=['mkv','mp4','avi'], desc='Video file'),
                                                    on_open=self._mx_media_pathdlg_on_open,
                                                    on_close=self._mx_media_pathdlg_on_close,
                                                    ).dispose_with(self)
                                                    
        elif new_source_type is self.SourceType.ImageSequence:
            self._mx_media_pathdlg = mx.PathState( config=mx.PathState.Config(dir_only=True, desc='Image Sequence'),
                                                    on_open=self._mx_media_pathdlg_on_open,
                                                    on_close=self._mx_media_pathdlg_on_close,
                                                    ).dispose_with(self)
        
        
        return new_source_type
        
    def _mx_media_pathdlg_on_close(self):
        print('_mx_media_pathdlg_on_close')
        self._tg.cancel_all()
        
        if self.mx_source_type.value is self.SourceType.VideoFile:
            
            self._vid_cap.release()
            self._vid_cap = None
        elif self.mx_source_type.value is self.SourceType.ImageSequence:
            self._files_paths = None
            
            
        self._mx_tl = self._mx_tl.dispose()
            
    def _mx_media_pathdlg_on_open(self, path : Path) -> bool:
        
        if self.mx_source_type.value is self.SourceType.VideoFile:
            vid_cap = cv2.VideoCapture()
            vid_cap.open(str(path), cv2.CAP_FFMPEG)          
            if vid_cap.isOpened(): 
                self._vid_cap = vid_cap
                
                
                frame_count=int(vid_cap.get(cv2.CAP_PROP_FRAME_COUNT))
                if frame_count != 0:
                    # fps=cap.get(cv2.CAP_PROP_FPS) )
                    mx_tl = self._mx_tl = RxTimeline(   frame_count=frame_count,
                                                        task_on_frame_idx = self._mx_tl_on_frame_idx  )
                    
                    self._mx_error.set(None)
                    return True
                else:
                    self._mx_error.set(f'No frames in {path}')
            else:
                self._mx_error.set(f'Unable to open {path}')
            
        elif self.mx_source_type.value is self.SourceType.ImageSequence:
            files_paths = self._files_paths = lib_path.get_files_paths(path, extensions=['.jpg','.png'])
            
            mx_tl = self._mx_tl = RxTimeline(   frame_count=len(files_paths),
                                                task_on_frame_idx = self._mx_tl_on_frame_idx  )
                                                        
            self._mx_error.set(None)
            return True
            
                    
        return False
   
    @ax.task
    def _mx_tl_on_frame_idx(self, frame_idx : int):
        yield ax.switch_to(self._sub_thread)
        
        if self.mx_source_type.value is self.SourceType.VideoFile:
            
            #print('_mx_tl_on_frame_idx', frame_idx)
            
            if not self._vid_cap.set(cv2.CAP_PROP_POS_FRAMES, frame_idx):            
                yield ax.switch_to(self._main_thread)
                yield ax.detach()
                self._mx_error.set(f'Error while seeking to {frame_idx} frame.')
                self._mx_media_pathdlg.close()
                return
            
            ret, frame = self._vid_cap.read()
            if not ret:
                yield ax.switch_to(self._main_thread)
                yield ax.detach()
                self._mx_error.set(f'Error while reading {frame_idx} frame.')
                self._mx_media_pathdlg.close()
                return
            
            yield ax.switch_to(self._main_thread)
            
        elif self.mx_source_type.value is self.SourceType.ImageSequence:
            filepath = self._files_paths[frame_idx]
            try:
                frame = lib_cv.imread(str(filepath))
            except Exception as e:
                yield ax.switch_to(self._main_thread)
                yield ax.detach()
                self._mx_error.set(f'Error while reading {frame_idx} frame from {filepath}. {str(e)}')
                self._mx_media_pathdlg.close()
                return
            
        self._mx_frame.set(frame)
          