import operator
import pickle
import threading
from collections import deque
from enum import Enum, auto
from pathlib import Path
from typing import Generator, Iterable, List, Sequence, Set, Union
from uuid import UUID

import cv2
import numpy as np

from xlib import caddys, ax, mx

#from .. import console as lib_con
# from .FMask import FMask
from .DFace import DFace
from .DImage import DImage

# from .UPerson import UPerson



class ESortBy(Enum):
    source_name = auto()
    blur = auto()

class RxDataset(mx.Disposable):
    """
    
    """
    class ClosedError(Exception): ...
                
    def __init__(self, path : Path, create_new : bool = False):
        """```
        
            create_new(False)   if True,    force recreates file
                                
                                if False,   creates new if file does not exist,
                                            otherwise opens file. 
                                            If file is not ds, closed state.
        
        raises NO errors.
        If error occurs, check mx_closed and mx_error.
        ```"""
        super().__init__()
        self._path = path
        self._create_new = create_new
        
        self._main_thread = ax.get_current_thread()
        self._ds_thread   = ax.Thread()
        self._tg          = ax.TaskGroup()
        self._tg_sort_by  = ax.TaskGroup()
        
        self._ds = None
        
        self._mx_initialized = mx.Flag(False).dispose_with(self)
        
        self._mx_closed = mx.Flag(False).dispose_with(self)
        self._mx_error = mx.Value[ Union[Exception, None] ](None).dispose_with(self)
        
        self._mx_face_count = mx.Number(0).dispose_with(self)
        self._mx_face_added = mx.Event1[DFace]().dispose_with(self)
        self._mx_face_removed = mx.Event1[DFace]().dispose_with(self)
        
        self._mx_face_sorted_update = mx.Event0().dispose_with(self)
        
        self._mx_image_count = mx.Number(0).dispose_with(self)
        self._mx_image_added = mx.Event1[DImage]().dispose_with(self)
        self._mx_image_removed = mx.Event1[DImage]().dispose_with(self)
        
        self._mx_sort_by_menu = mx.Menu[ESortBy]([ e for e in ESortBy], control=self.sort_faces_by).dispose_with(self)
        
        
        self._initialization_task()    
    
    @ax.task
    def _initialization_task(self):
        yield ax.attach_to(self._tg)
        yield ax.switch_to(self._ds_thread)
        
        err = None
        try:
            ds = caddys.File(self._path, create_new=self._create_new)
            
            faces_uuids = ds.get('faces_uuids', set())
            ds.mount('faces_uuids', faces_uuids)
            
            self._faces_uuids : Set[UUID] = faces_uuids  # thread-safe read via tuple(), change : ds_thread
            self._faces_uuids_sorted = list(faces_uuids) # thread-safe read, change : ds_thread
            self._faces_write_lock = threading.Lock()
            # ^ Tasks concurrent for _faces_write_lock:
            #   add_face , sort_faces_by
            
            face_count = len(faces_uuids)
            
        except Exception as e:
            err = e
            self._ds_close()
        
        if err is None:  
            self._ds = ds
                
        yield ax.switch_to(self._main_thread)
                
        if err is not None:
            self.mx_error.set(err)
            self.mx_closed.set(True)   
            self._mx_initialized.set(True)
            return
                    
        self._mx_face_count.set(face_count)
        self._mx_initialized.set(True)
        
        self._auto_flush_task()
    
    def dispose(self):
        super().dispose()
        self._tg = self._tg.dispose()
        self._tg_sort_by = self._tg_sort_by.dispose()
        
        self._ds_thread = self._ds_thread.dispose()
        self._ds_flush()
        self._ds_close()
    
    @property
    def mx_initialized(self) -> mx.Flag: 
        """Indicates that RxDataset is initialized, and mx_closed/mx_error can be read."""
        return self._mx_closed
        
    @property
    def mx_closed(self) -> mx.Flag: 
        """
        Indicates that RxDataset is closed and will no longer interact to commands.
        Avail only mx_initialized is True
        """
        return self._mx_closed
    @property
    def mx_error(self) -> mx.Value[ Union[Exception, None] ]: 
        """
        Optional error if closed.
        Avail only mx_initialized is True
        """
        return self._mx_error
                 
    @property    
    def mx_face_count(self) -> mx.Number: return self._mx_face_count
    @property    
    def mx_face_added(self) -> mx.Event1[DFace]: return self._mx_face_added
    @property    
    def mx_face_removed(self) -> mx.Event1[DFace]: return self._mx_face_removed
    @property    
    def mx_face_sorted_update(self) -> mx.Event0: 
        """Signal emitted when sorted faces has been updated"""
        return self._mx_face_sorted_update
    
    @property    
    def mx_image_count(self) -> mx.Number: return self._mx_image_count
    @property    
    def mx_image_added(self) -> mx.Event1[DImage]: return self._mx_image_added
    @property    
    def mx_image_removed(self) -> mx.Event1[DImage]: return self._mx_image_removed
    @property    
    def mx_sort_by_menu(self) -> mx.Menu[ESortBy]: return self._mx_sort_by_menu
    
   
       
                
    ######################
    ### Common
    ######################
    @ax.task
    def flush(self) -> str:
        """Writes pending data."""
        yield ax.attach_to(self._tg)
        yield ax.switch_to(self._ds_thread)
        
        err = None
        try:
            if self._ds is None:
                raise self.ClosedError()
            self._ds_flush()            
        except Exception as e:
            err=e
            self._ds_close()
            
        yield ax.switch_to(self._main_thread)

        if err is not None:
            self._mx_error.set(err)
            self._mx_closed.set(True)     
            yield ax.cancel(err)     
             
    @ax.task
    def _auto_flush_task(self):
        yield ax.attach_to(self._tg)
        
        while True:
            yield ax.sleep(10.0)
            
            t = self.flush()
            yield ax.wait(t)
            if not t.succeeded:
                break
        
    ######################
    ### DFace
    ######################
    def get_faces_uuid(self) -> Sequence[UUID]:
        """Returns sequence of available faces uuid"""
        return tuple(self._faces_uuids)
        
    def get_face_count(self) -> int:
        return len(self._faces_uuids)
        
            
    @ax.task
    def add_face(self, dface_or_list : Union[DFace, Iterable[DFace]], ):
        """adds or updates existing DFace in DS"""
        yield ax.attach_to(self._tg)
        
        yield ax.switch_to(self._main_thread)
        if self._mx_closed.value:
            yield ax.cancel()
            
        yield ax.switch_to(self._ds_thread)
        
        while not self._faces_write_lock.acquire(blocking=False):
            yield ax.sleep(0)
            
        try:
            err = None
            try:
                ds = self._ds
                if ds is None:
                    raise self.ClosedError()
        
                if not isinstance(dface_or_list, Iterable):
                    dface_or_list : List[DFace] = [dface_or_list]

                for dface in dface_or_list:
                    ds.collect(dface.uuid, {'uuid' : dface.uuid,
                                            'image_uuid' : dface.image_uuid,
                                            'source_name' : dface.source_name})
                    
                    self._faces_uuids.add(dface.uuid)
                    self._faces_uuids_sorted.append(dface.uuid)
            
            except ax.TaskFinishError as e:
                raise e
            except Exception as e:
                err=e
                self._ds_close()
            
            yield ax.switch_to(self._main_thread)
            
            if err is None:
                for dface in dface_or_list:           
                    print('self._mx_face_count.value', self._mx_face_count.value) 
                    self._mx_face_count.set(self._mx_face_count.value+1)
                    self._mx_face_added.push(dface)
            else:
                self._mx_error.set(err)
                self._mx_closed.set(True)     
                yield ax.cancel(err)     
                
        except ax.TaskFinishError:
            ...
        
        self._faces_write_lock.release()
        
       
            
    @ax.task
    def get_face_by_uuid(self, uuid : UUID) -> Union[DFace, None]:
        """
        returns DFace by uuid if available.
        """
        yield ax.attach_to(self._tg)
        yield ax.switch_to(self._ds_thread)
        
        dface = None
        err = None
        try:
            ds = self._ds
            if ds is None:
                raise self.ClosedError()
            
            if uuid in self._faces_uuids:
                if (dface_state := ds.get(uuid, None)) is not None:
                    dface = DFace()
                    dface.uuid = dface_state['uuid']
                    dface.image_uuid = dface_state.get('image_uuid', None)
                    dface.source_name = dface_state.get('source_name', None)
            
        except Exception as e:
            err=e
            self._ds_close()
        
        yield ax.switch_to(self._main_thread)
         
        if err is None:
            return dface
        else:
            self._mx_error.set(err)
            self._mx_closed.set(True)     
            yield ax.cancel(err)   
       
    @ax.task
    def get_face_by_sorted_id(self, id : int) -> Union[DFace, None]:
        """"""
        yield ax.attach_to(self._tg)
        print('get_face_by_sorted_id', id, len(self._faces_uuids_sorted))
        uuid = self._faces_uuids_sorted[id]
        yield ax.propagate( self.get_face_by_uuid(uuid) )
        
    @ax.task
    def sort_faces_by(self, sort_by : ESortBy):
        """"""        
        yield ax.attach_to(self._tg_sort_by, cancel_all=True)
        yield ax.attach_to(self._tg)
        yield ax.switch_to(self._main_thread)
        
        while not self._faces_write_lock.acquire(blocking=False):
            yield ax.sleep(0)
        
        try:
            # self._mx_sort_by_progress_bar.start()
            
            yield ax.switch_to(self._ds_thread)
            
            s = deque()
            for face_uuid in self._faces_uuids:
                
                face_t = self.get_face_by_uuid(face_uuid)
                yield ax.wait(face_t)
                
                if not face_t.succeeded:
                    yield ax.cancel(face_t.error)
                    
                dface = face_t.result
                if dface is None:
                    yield ax.cancel()
                    
                if sort_by == ESortBy.source_name:
                    cond = dface.source_name
                    if cond is None:
                        cond = dface.uuid.hex
                        
                    s.append( (face_uuid, cond) )
                
                #self.get_image_by_uuid(image_uuid)
             
            if sort_by == ESortBy.source_name:   
                s = sorted(s, key=operator.itemgetter(1), reverse=False)
                s = [ x[0] for x in s ]
                
            self._faces_uuids_sorted = s
            
            yield ax.switch_to(self._main_thread)
            
            self._mx_face_sorted_update.push()        
            
        except ax.TaskFinishError: 
            ...
            
        self._faces_write_lock.release()
                
        
        
    # @ax.task
    # def get_faces(self) -> ax.Task[ List[DFace] ]:
    #     if not self._mx_initialized.value:
    #         raise Exception('Not initialized')
            
    #     yield ax.attach_to(self._tg)
    #     yield ax.switch_to(self._ds_thread)
        
    #     faces = []
    #     with self._db_lock:
    #         keys = self._DFace_grp.keys()
        
    #     for key in keys:
    #         face_task = self.get_face( UUID(key) )
    #         yield ax.wait(face_task)
    #         if face_task.succeeded:
    #             faces.append(face_task.result)
    #         else:
    #             yield ax.cancel(face_task.)
    
    #     return faces       
        
    # ###################
    # ### DImage
    # ###################
    # @ax.task
    # def add_DImage(self,    dimage_or_list : Union[DImage, Iterable[DImage]], 
    #                         quality : int = -1) -> ax.Task:
    #     """
    #     adds or updates existing DImage in DB

    #      uimage       DImage or list
         
    #      quality(100)   for float32 images:
    #                         always saved lossless
    #                     for uint8 images:
    #                         -1 is lossless
    #                         0-100 jpeg2000 codec is used
    #     """
    #     if not self._mx_initialized.value:
    #         raise Exception('Not initialized')
            
    #     if quality < -1 or quality > 100:
    #         raise ValueError('quality must be in range [-1..100]')
            
    #     if not isinstance(dimage_or_list, Iterable):
    #         dimage_or_list : List[DImage] = [dimage_or_list]

    #     yield ax.attach_to(self._tg)
        
    #     for dimage in dimage_or_list:
    #         yield ax.switch_to(self._ds_thread)

    #         imencode_args = []
    #         np_image = dimage.np_image
    #         if np_image is not None:
    #             np_image_dtype = np_image.dtype
    #             if np_image_dtype == np.float32:
    #                 format = '.tif'
    #             elif np_image_dtype == np.uint8:
    #                 if quality == -1:
    #                     format = '.tif'
    #                 else:
    #                     imencode_args = [int(cv2.IMWRITE_JPEG2000_COMPRESSION_X1000), quality*10]
    #                     format = '.jp2'
    #             else:
    #                 raise ValueError('np_image.dtype must be in [np.uint8, np.float32]')
    
    #             ret, data_bytes = cv2.imencode(format, np_image, imencode_args)
    #             if not ret:
    #                 raise Exception(f'Unable to encode image format {format}')

    #         key = dimage.uuid.hex
            
    #         state = {
    #             'uuid' : dimage.uuid,
    #             'name' : dimage.name,
    #         }

    #         self._locked_group_write_bytes(self._DImage_grp, key, pickle.dumps(state) )
            
    #         if np_image is not None:
    #             d = self._locked_group_write_bytes(self._DImage_np_image_grp, key, data_bytes.data)
    #             d.attrs['format'] = format
    #             d.attrs['quality'] = quality
            
    #         #print(self._DImage_grp.keys())
    #         #import code
    #         #code.interact(local=dict(globals(), **locals()))

    #         yield ax.switch_to(self._main_thread)
            
    #         self._mx_image_count.set(self._mx_image_count.value+1)
    #         self._mx_image_added.push(dimage)

    # def get_DImage_count(self) -> int:
    #     with self._db_lock:
    #         return len(self._DImage_grp.keys())
        
    # @ax.task
    # def get_DImage(self, uuid : UUID) -> Union[DImage, None]:
    #     if not self._mx_initialized.value:
    #         raise Exception('Not initialized')
            
    #     yield ax.attach_to(self._tg)
    #     yield ax.switch_to(self._ds_thread)
        
    #     key = uuid.hex
    #     data = self._locked_group_read_bytes(self._DImage_grp, key)
    #     if data is not None:
            
    #         import code
    #         code.interact(local=dict(globals(), **locals()))

    #         # uimg = DImage.from_state(pickle.loads(data))

    #         # image_data = self._locked_group_read_bytes(self._DImage_image_data_grp, key)
    #         # if image_data is not None:
    #         #     uimg.assign_image (cv2.imdecode(np.frombuffer(image_data, dtype=np.uint8), flags=cv2.IMREAD_UNCHANGED))
    #         return None
    #         return uimg  
    #     return None
     
   
    # def get_DFace_by_uuid(self, uuid : bytes) -> Union[DFace, None]:
    #     data = self._locked_group_read_bytes(self._DFace_grp, uuid.hex())
    #     if data is None:
    #         return None
    #     return DFace.from_state(pickle.loads(data))
    
    ####################
    ### internal
    ####################
    def _ds_flush(self):
        if self._ds is not None:
            self._ds.flush()
                
    def _ds_close(self):
        if self._ds is not None:
            self._ds.close()
            self._ds = None           
            
    def __repr__(self): return self.__str__()
    def __str__(self):
        
        return f"{super().__str__()}[RxDataset]"# DImage:{self.get_DImage_count()} DFace:{self.get_DFace_count()} UPerson:{self.get_UPerson_count()}"

   
    


    # def get_all_DImage(self) -> List[DImage]:
    #     return [ self._get_DImage_by_key(key) for key in self._DImage_grp.keys() ]
        
    # def get_all_DImage_uuids(self) -> List[ uuid.UUID ]:
    #     return [ uuid.UUID(key).bytes for key in self._DImage_grp.keys() ]
        


    # def get_DImage_by_uuid(self, uuid : bytes) -> Union[DImage, None]:
    #     return self._get_DImage_by_key(uuid.hex())

    # def delete_DImage_by_uuid(self, uuid : bytes):
    #     key = uuid.hex()
    #     if key in self._DImage_grp:
    #         del self._DImage_grp[key]
    #     if key in self._DImage_image_data_grp:
    #         del self._DImage_image_data_grp[key]

    # def iter_DImage(self, include_key=False) -> Generator[DImage, None, None]:
    #     """
    #     returns Generator of DImage
    #     """
    #     for key in self._DImage_grp.keys():
    #         uimg = self._get_DImage_by_key(key, check_key=False)
    #         yield (uimg, key) if include_key else uimg

    # def delete_all_DImage(self):
    #     """
    #     deletes all DImage from DB
    #     """
    #     for key in self._DImage_grp.keys():
    #         del self._DImage_grp[key]
    #     for key in self._DImage_image_data_grp.keys():
    #         del self._DImage_image_data_grp[key]


   
    # def delete_DFace_by_uuid(self, uuid : bytes) -> bool:
    #     key = uuid.hex()
    #     if key in self._DFace_grp:
    #         del self._DFace_grp[key]
    #         return True
    #     return False
    
           
    # def iter_DFace(self) -> Generator[DFace, None, None]:
    #     """
    #     returns Generator of DFace
    #     """
    #     for key in self._DFace_grp.keys():
    #         yield DFace.from_state(pickle.loads(self._locked_group_read_bytes(self._DFace_grp, key, check_key=False)))

    # def delete_all_DFace(self):
    #     """
    #     deletes all DFace from DB
    #     """
    #     for key in self._DFace_grp.keys():
    #         del self._DFace_grp[key]

    # ###################
    # ### UPerson
    # ###################

    # def add_UPerson(self, uperson_or_list : UPerson, update_existing=True):
    #     """
    #     add or update UPerson in DB
    #     """
    #     if not isinstance(uperson_or_list, Iterable):
    #         uperson_or_list : List[UPerson] = [uperson_or_list]

    #     for uperson in uperson_or_list:
    #         self._locked_group_write_bytes(self._UPerson_grp, uperson.get_uuid().hex(), pickle.dumps(uperson.dump_state()), update_existing=update_existing )

    # def get_UPerson_count(self) -> int:
    #     return len(self._UPerson_grp.keys())

    # def get_all_UPerson(self) -> List[UPerson]:
    #     return [ UPerson.from_state(pickle.loads(self._locked_group_read_bytes(self._UPerson_grp, key, check_key=False))) for key in self._UPerson_grp.keys() ]
    
    # def get_all_UPerson_uuids(self) -> List[bytes]:
    #     return [ uuid.UUID(key).bytes for key in self._UPerson_grp.keys() ]
    
    # def get_UPerson_by_uuid(self, uuid : bytes) -> Union[UPerson, None]:
    #     data = self._locked_group_read_bytes(self._UPerson_grp, uuid.hex())
    #     if data is None:
    #         return None
    #     return UPerson.from_state(pickle.loads(data))

    # def delete_UPerson_by_uuid(self, uuid : bytes) -> bool:
    #     key = uuid.hex()
    #     if key in self._UPerson_grp:
    #         del self._UPerson_grp[key]
    #         return True
    #     return False

    # def iter_UPerson(self) -> Generator[UPerson, None, None]:
    #     """
    #     returns Generator of UPerson
    #     """
    #     for key in self._UPerson_grp.keys():
    #         yield UPerson.from_state(pickle.loads(self._locked_group_read_bytes(self._UPerson_grp, key, check_key=False)))

    # def delete_all_UPerson(self):
    #     """
    #     deletes all UPerson from DB
    #     """
    #     for key in self._UPerson_grp.keys():
    #         del self._UPerson_grp[key]

    

    # def optimize(self, verbose=True):
    #     """
    #     recreate Dataset with optimized structure.
    #     """
    #     if verbose:
    #         print(f'Optimizing {self._path.name}...')

    #     tmp_path = self._path.parent / (self._path.stem + '_optimizing' + self._path.suffix)

    #     tmp_fs = Dataset(tmp_path, write_access=True, recreate=True)
    #     self._locked_group_copy(tmp_fs._DFace_grp, self._DFace_grp, verbose=verbose)
    #     self._locked_group_copy(tmp_fs._UPerson_grp, self._UPerson_grp, verbose=verbose)
    #     self._locked_group_copy(tmp_fs._DImage_grp, self._DImage_grp, verbose=verbose)
    #     self._locked_group_copy(tmp_fs._DImage_image_data_grp, self._DImage_image_data_grp, verbose=verbose)
    #     tmp_fs._close()

    #     self._close()
    #     self._path.unlink()
    #     tmp_path.rename(self._path)
    #     self._open()
    