/*
 * Decompiled with CFR 0.152.
 */
package android.util.apk;

import android.util.ArrayMap;
import android.util.Pair;
import android.util.apk.ByteBufferDataSource;
import android.util.apk.ByteBufferFactory;
import android.util.apk.DataDigester;
import android.util.apk.DataSource;
import android.util.apk.MemoryMappedFileDataSource;
import android.util.apk.SignatureInfo;
import android.util.apk.SignatureNotFoundException;
import android.util.apk.VerityBuilder;
import android.util.apk.ZipUtils;
import java.io.FileDescriptor;
import java.io.IOException;
import java.io.RandomAccessFile;
import java.nio.BufferUnderflowException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.security.DigestException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.MGF1ParameterSpec;
import java.security.spec.PSSParameterSpec;
import java.util.Arrays;
import java.util.Iterator;
import java.util.Map;

class ApkSigningBlockUtils {
    private static final int CHUNK_SIZE_BYTES = 0x100000;
    static final int SIGNATURE_RSA_PSS_WITH_SHA256 = 257;
    static final int SIGNATURE_RSA_PSS_WITH_SHA512 = 258;
    static final int SIGNATURE_RSA_PKCS1_V1_5_WITH_SHA256 = 259;
    static final int SIGNATURE_RSA_PKCS1_V1_5_WITH_SHA512 = 260;
    static final int SIGNATURE_ECDSA_WITH_SHA256 = 513;
    static final int SIGNATURE_ECDSA_WITH_SHA512 = 514;
    static final int SIGNATURE_DSA_WITH_SHA256 = 769;
    static final int SIGNATURE_VERITY_RSA_PKCS1_V1_5_WITH_SHA256 = 1057;
    static final int SIGNATURE_VERITY_ECDSA_WITH_SHA256 = 1059;
    static final int SIGNATURE_VERITY_DSA_WITH_SHA256 = 1061;
    static final int CONTENT_DIGEST_CHUNKED_SHA256 = 1;
    static final int CONTENT_DIGEST_CHUNKED_SHA512 = 2;
    static final int CONTENT_DIGEST_VERITY_CHUNKED_SHA256 = 3;
    static final int CONTENT_DIGEST_SHA256 = 4;
    private static final int[] V4_CONTENT_DIGEST_ALGORITHMS = new int[]{2, 3, 1};
    private static final long APK_SIG_BLOCK_MAGIC_HI = 3617552046287187010L;
    private static final long APK_SIG_BLOCK_MAGIC_LO = 2334950737559900225L;
    private static final int APK_SIG_BLOCK_MIN_SIZE = 32;

    private ApkSigningBlockUtils() {
    }

    static SignatureInfo findSignature(RandomAccessFile apk, int blockId) throws IOException, SignatureNotFoundException {
        Pair<ByteBuffer, Long> eocdAndOffsetInFile = ApkSigningBlockUtils.getEocd(apk);
        ByteBuffer eocd = (ByteBuffer)eocdAndOffsetInFile.first;
        long eocdOffset = (Long)eocdAndOffsetInFile.second;
        if (ZipUtils.isZip64EndOfCentralDirectoryLocatorPresent(apk, eocdOffset)) {
            throw new SignatureNotFoundException("ZIP64 APK not supported");
        }
        long centralDirOffset = ApkSigningBlockUtils.getCentralDirOffset(eocd, eocdOffset);
        Pair<ByteBuffer, Long> apkSigningBlockAndOffsetInFile = ApkSigningBlockUtils.findApkSigningBlock(apk, centralDirOffset);
        ByteBuffer apkSigningBlock = (ByteBuffer)apkSigningBlockAndOffsetInFile.first;
        long apkSigningBlockOffset = (Long)apkSigningBlockAndOffsetInFile.second;
        ByteBuffer apkSignatureSchemeBlock = ApkSigningBlockUtils.findApkSignatureSchemeBlock(apkSigningBlock, blockId);
        return new SignatureInfo(apkSignatureSchemeBlock, apkSigningBlockOffset, centralDirOffset, eocdOffset, eocd);
    }

    static void verifyIntegrity(Map<Integer, byte[]> expectedDigests, RandomAccessFile apk, SignatureInfo signatureInfo) throws SecurityException {
        if (expectedDigests.isEmpty()) {
            throw new SecurityException("No digests provided");
        }
        boolean neverVerified = true;
        ArrayMap<Integer, byte[]> expected1MbChunkDigests = new ArrayMap<Integer, byte[]>();
        if (expectedDigests.containsKey(1)) {
            expected1MbChunkDigests.put(1, expectedDigests.get(1));
        }
        if (expectedDigests.containsKey(2)) {
            expected1MbChunkDigests.put(2, expectedDigests.get(2));
        }
        if (!expected1MbChunkDigests.isEmpty()) {
            try {
                ApkSigningBlockUtils.verifyIntegrityFor1MbChunkBasedAlgorithm(expected1MbChunkDigests, apk.getFD(), signatureInfo);
                neverVerified = false;
            }
            catch (IOException e) {
                throw new SecurityException("Cannot get FD", e);
            }
        }
        if (expectedDigests.containsKey(3)) {
            ApkSigningBlockUtils.verifyIntegrityForVerityBasedAlgorithm(expectedDigests.get(3), apk, signatureInfo);
            neverVerified = false;
        }
        if (neverVerified) {
            throw new SecurityException("No known digest exists for integrity check");
        }
    }

    static boolean isSupportedSignatureAlgorithm(int sigAlgorithm) {
        switch (sigAlgorithm) {
            case 257: 
            case 258: 
            case 259: 
            case 260: 
            case 513: 
            case 514: 
            case 769: 
            case 1057: 
            case 1059: 
            case 1061: {
                return true;
            }
        }
        return false;
    }

    private static void verifyIntegrityFor1MbChunkBasedAlgorithm(Map<Integer, byte[]> expectedDigests, FileDescriptor apkFileDescriptor, SignatureInfo signatureInfo) throws SecurityException {
        byte[][] actualDigests;
        MemoryMappedFileDataSource beforeApkSigningBlock = new MemoryMappedFileDataSource(apkFileDescriptor, 0L, signatureInfo.apkSigningBlockOffset);
        MemoryMappedFileDataSource centralDir = new MemoryMappedFileDataSource(apkFileDescriptor, signatureInfo.centralDirOffset, signatureInfo.eocdOffset - signatureInfo.centralDirOffset);
        ByteBuffer eocdBuf = signatureInfo.eocd.duplicate();
        eocdBuf.order(ByteOrder.LITTLE_ENDIAN);
        ZipUtils.setZipEocdCentralDirectoryOffset(eocdBuf, signatureInfo.apkSigningBlockOffset);
        ByteBufferDataSource eocd = new ByteBufferDataSource(eocdBuf);
        int[] digestAlgorithms = new int[expectedDigests.size()];
        int digestAlgorithmCount = 0;
        Iterator<Integer> iterator = expectedDigests.keySet().iterator();
        while (iterator.hasNext()) {
            int digestAlgorithm;
            digestAlgorithms[digestAlgorithmCount] = digestAlgorithm = iterator.next().intValue();
            ++digestAlgorithmCount;
        }
        try {
            actualDigests = ApkSigningBlockUtils.computeContentDigestsPer1MbChunk(digestAlgorithms, new DataSource[]{beforeApkSigningBlock, centralDir, eocd});
        }
        catch (DigestException e) {
            throw new SecurityException("Failed to compute digest(s) of contents", e);
        }
        for (int i = 0; i < digestAlgorithms.length; ++i) {
            byte[] actualDigest;
            int digestAlgorithm = digestAlgorithms[i];
            byte[] expectedDigest = expectedDigests.get(digestAlgorithm);
            if (MessageDigest.isEqual(expectedDigest, actualDigest = actualDigests[i])) continue;
            throw new SecurityException(ApkSigningBlockUtils.getContentDigestAlgorithmJcaDigestAlgorithm(digestAlgorithm) + " digest of contents did not verify");
        }
    }

    private static byte[][] computeContentDigestsPer1MbChunk(int[] digestAlgorithms, DataSource[] contents) throws DigestException {
        long totalChunkCountLong = 0L;
        for (DataSource input : contents) {
            totalChunkCountLong += ApkSigningBlockUtils.getChunkCount(input.size());
        }
        if (totalChunkCountLong >= 0x1FFFFFL) {
            throw new DigestException("Too many chunks: " + totalChunkCountLong);
        }
        int totalChunkCount = (int)totalChunkCountLong;
        byte[][] digestsOfChunks = new byte[digestAlgorithms.length][];
        for (int i = 0; i < digestAlgorithms.length; ++i) {
            int digestAlgorithm = digestAlgorithms[i];
            int digestOutputSizeBytes = ApkSigningBlockUtils.getContentDigestAlgorithmOutputSizeBytes(digestAlgorithm);
            byte[] concatenationOfChunkCountAndChunkDigests = new byte[5 + totalChunkCount * digestOutputSizeBytes];
            concatenationOfChunkCountAndChunkDigests[0] = 90;
            ApkSigningBlockUtils.setUnsignedInt32LittleEndian(totalChunkCount, concatenationOfChunkCountAndChunkDigests, 1);
            digestsOfChunks[i] = concatenationOfChunkCountAndChunkDigests;
        }
        byte[] chunkContentPrefix = new byte[5];
        chunkContentPrefix[0] = -91;
        int chunkIndex = 0;
        MessageDigest[] mds = new MessageDigest[digestAlgorithms.length];
        for (int i = 0; i < digestAlgorithms.length; ++i) {
            String jcaAlgorithmName = ApkSigningBlockUtils.getContentDigestAlgorithmJcaDigestAlgorithm(digestAlgorithms[i]);
            try {
                mds[i] = MessageDigest.getInstance(jcaAlgorithmName);
                continue;
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(jcaAlgorithmName + " digest not supported", e);
            }
        }
        MultipleDigestDataDigester digester = new MultipleDigestDataDigester(mds);
        int dataSourceIndex = 0;
        for (DataSource dataSource : contents) {
            long inputOffset = 0L;
            long inputRemaining = dataSource.size();
            while (inputRemaining > 0L) {
                int i;
                int chunkSize = (int)Math.min(inputRemaining, 0x100000L);
                ApkSigningBlockUtils.setUnsignedInt32LittleEndian(chunkSize, chunkContentPrefix, 1);
                for (i = 0; i < mds.length; ++i) {
                    mds[i].update(chunkContentPrefix);
                }
                try {
                    dataSource.feedIntoDataDigester(digester, inputOffset, chunkSize);
                }
                catch (IOException e) {
                    throw new DigestException("Failed to digest chunk #" + chunkIndex + " of section #" + dataSourceIndex, e);
                }
                for (i = 0; i < digestAlgorithms.length; ++i) {
                    MessageDigest md = mds[i];
                    byte[] concatenationOfChunkCountAndChunkDigests = digestsOfChunks[i];
                    int digestAlgorithm = digestAlgorithms[i];
                    int expectedDigestSizeBytes = ApkSigningBlockUtils.getContentDigestAlgorithmOutputSizeBytes(digestAlgorithm);
                    int actualDigestSizeBytes = md.digest(concatenationOfChunkCountAndChunkDigests, 5 + chunkIndex * expectedDigestSizeBytes, expectedDigestSizeBytes);
                    if (actualDigestSizeBytes == expectedDigestSizeBytes) continue;
                    throw new RuntimeException("Unexpected output size of " + md.getAlgorithm() + " digest: " + actualDigestSizeBytes);
                }
                inputOffset += (long)chunkSize;
                inputRemaining -= (long)chunkSize;
                ++chunkIndex;
            }
            ++dataSourceIndex;
        }
        byte[][] result = new byte[digestAlgorithms.length][];
        for (int i = 0; i < digestAlgorithms.length; ++i) {
            MessageDigest md;
            int digestAlgorithm = digestAlgorithms[i];
            byte[] byArray = digestsOfChunks[i];
            String jcaAlgorithmName = ApkSigningBlockUtils.getContentDigestAlgorithmJcaDigestAlgorithm(digestAlgorithm);
            try {
                md = MessageDigest.getInstance(jcaAlgorithmName);
            }
            catch (NoSuchAlgorithmException e) {
                throw new RuntimeException(jcaAlgorithmName + " digest not supported", e);
            }
            byte[] output = md.digest(byArray);
            result[i] = output;
        }
        return result;
    }

    static byte[] parseVerityDigestAndVerifySourceLength(byte[] data, long fileSize, SignatureInfo signatureInfo) throws SecurityException {
        int kRootHashSize = 32;
        int kSourceLengthSize = 8;
        if (data.length != kRootHashSize + kSourceLengthSize) {
            throw new SecurityException("Verity digest size is wrong: " + data.length);
        }
        ByteBuffer buffer = ByteBuffer.wrap(data).order(ByteOrder.LITTLE_ENDIAN);
        buffer.position(kRootHashSize);
        long expectedSourceLength = buffer.getLong();
        long signingBlockSize = signatureInfo.centralDirOffset - signatureInfo.apkSigningBlockOffset;
        if (expectedSourceLength != fileSize - signingBlockSize) {
            throw new SecurityException("APK content size did not verify");
        }
        return Arrays.copyOfRange(data, 0, kRootHashSize);
    }

    private static void verifyIntegrityForVerityBasedAlgorithm(byte[] expectedDigest, RandomAccessFile apk, SignatureInfo signatureInfo) throws SecurityException {
        try {
            byte[] expectedRootHash = ApkSigningBlockUtils.parseVerityDigestAndVerifySourceLength(expectedDigest, apk.length(), signatureInfo);
            VerityBuilder.VerityResult verity = VerityBuilder.generateApkVerityTree(apk, signatureInfo, new ByteBufferFactory(){

                @Override
                public ByteBuffer create(int capacity) {
                    return ByteBuffer.allocate(capacity);
                }
            });
            if (!Arrays.equals(expectedRootHash, verity.rootHash)) {
                throw new SecurityException("APK verity digest of contents did not verify");
            }
        }
        catch (IOException | DigestException | NoSuchAlgorithmException e) {
            throw new SecurityException("Error during verification", e);
        }
    }

    static Pair<ByteBuffer, Long> getEocd(RandomAccessFile apk) throws IOException, SignatureNotFoundException {
        Pair<ByteBuffer, Long> eocdAndOffsetInFile = ZipUtils.findZipEndOfCentralDirectoryRecord(apk);
        if (eocdAndOffsetInFile == null) {
            throw new SignatureNotFoundException("Not an APK file: ZIP End of Central Directory record not found");
        }
        return eocdAndOffsetInFile;
    }

    static long getCentralDirOffset(ByteBuffer eocd, long eocdOffset) throws SignatureNotFoundException {
        long centralDirOffset = ZipUtils.getZipEocdCentralDirectoryOffset(eocd);
        if (centralDirOffset > eocdOffset) {
            throw new SignatureNotFoundException("ZIP Central Directory offset out of range: " + centralDirOffset + ". ZIP End of Central Directory offset: " + eocdOffset);
        }
        long centralDirSize = ZipUtils.getZipEocdCentralDirectorySizeBytes(eocd);
        if (centralDirOffset + centralDirSize != eocdOffset) {
            throw new SignatureNotFoundException("ZIP Central Directory is not immediately followed by End of Central Directory");
        }
        return centralDirOffset;
    }

    private static long getChunkCount(long inputSizeBytes) {
        return (inputSizeBytes + 0x100000L - 1L) / 0x100000L;
    }

    static int compareSignatureAlgorithm(int sigAlgorithm1, int sigAlgorithm2) {
        int digestAlgorithm1 = ApkSigningBlockUtils.getSignatureAlgorithmContentDigestAlgorithm(sigAlgorithm1);
        int digestAlgorithm2 = ApkSigningBlockUtils.getSignatureAlgorithmContentDigestAlgorithm(sigAlgorithm2);
        return ApkSigningBlockUtils.compareContentDigestAlgorithm(digestAlgorithm1, digestAlgorithm2);
    }

    private static int compareContentDigestAlgorithm(int digestAlgorithm1, int digestAlgorithm2) {
        switch (digestAlgorithm1) {
            case 1: {
                switch (digestAlgorithm2) {
                    case 1: {
                        return 0;
                    }
                    case 2: 
                    case 3: {
                        return -1;
                    }
                }
                throw new IllegalArgumentException("Unknown digestAlgorithm2: " + digestAlgorithm2);
            }
            case 2: {
                switch (digestAlgorithm2) {
                    case 1: 
                    case 3: {
                        return 1;
                    }
                    case 2: {
                        return 0;
                    }
                }
                throw new IllegalArgumentException("Unknown digestAlgorithm2: " + digestAlgorithm2);
            }
            case 3: {
                switch (digestAlgorithm2) {
                    case 2: {
                        return -1;
                    }
                    case 3: {
                        return 0;
                    }
                    case 1: {
                        return 1;
                    }
                }
                throw new IllegalArgumentException("Unknown digestAlgorithm2: " + digestAlgorithm2);
            }
        }
        throw new IllegalArgumentException("Unknown digestAlgorithm1: " + digestAlgorithm1);
    }

    static int getSignatureAlgorithmContentDigestAlgorithm(int sigAlgorithm) {
        switch (sigAlgorithm) {
            case 257: 
            case 259: 
            case 513: 
            case 769: {
                return 1;
            }
            case 258: 
            case 260: 
            case 514: {
                return 2;
            }
            case 1057: 
            case 1059: 
            case 1061: {
                return 3;
            }
        }
        throw new IllegalArgumentException("Unknown signature algorithm: 0x" + Long.toHexString(sigAlgorithm & 0xFFFFFFFF));
    }

    static String getContentDigestAlgorithmJcaDigestAlgorithm(int digestAlgorithm) {
        switch (digestAlgorithm) {
            case 1: 
            case 3: {
                return "SHA-256";
            }
            case 2: {
                return "SHA-512";
            }
        }
        throw new IllegalArgumentException("Unknown content digest algorthm: " + digestAlgorithm);
    }

    private static int getContentDigestAlgorithmOutputSizeBytes(int digestAlgorithm) {
        switch (digestAlgorithm) {
            case 1: 
            case 3: {
                return 32;
            }
            case 2: {
                return 64;
            }
        }
        throw new IllegalArgumentException("Unknown content digest algorthm: " + digestAlgorithm);
    }

    static String getSignatureAlgorithmJcaKeyAlgorithm(int sigAlgorithm) {
        switch (sigAlgorithm) {
            case 257: 
            case 258: 
            case 259: 
            case 260: 
            case 1057: {
                return "RSA";
            }
            case 513: 
            case 514: 
            case 1059: {
                return "EC";
            }
            case 769: 
            case 1061: {
                return "DSA";
            }
        }
        throw new IllegalArgumentException("Unknown signature algorithm: 0x" + Long.toHexString(sigAlgorithm & 0xFFFFFFFF));
    }

    static Pair<String, ? extends AlgorithmParameterSpec> getSignatureAlgorithmJcaSignatureAlgorithm(int sigAlgorithm) {
        switch (sigAlgorithm) {
            case 257: {
                return Pair.create("SHA256withRSA/PSS", new PSSParameterSpec("SHA-256", "MGF1", MGF1ParameterSpec.SHA256, 32, 1));
            }
            case 258: {
                return Pair.create("SHA512withRSA/PSS", new PSSParameterSpec("SHA-512", "MGF1", MGF1ParameterSpec.SHA512, 64, 1));
            }
            case 259: 
            case 1057: {
                return Pair.create("SHA256withRSA", null);
            }
            case 260: {
                return Pair.create("SHA512withRSA", null);
            }
            case 513: 
            case 1059: {
                return Pair.create("SHA256withECDSA", null);
            }
            case 514: {
                return Pair.create("SHA512withECDSA", null);
            }
            case 769: 
            case 1061: {
                return Pair.create("SHA256withDSA", null);
            }
        }
        throw new IllegalArgumentException("Unknown signature algorithm: 0x" + Long.toHexString(sigAlgorithm & 0xFFFFFFFF));
    }

    static byte[] pickBestDigestForV4(Map<Integer, byte[]> contentDigests) {
        for (int algo : V4_CONTENT_DIGEST_ALGORITHMS) {
            if (!contentDigests.containsKey(algo)) continue;
            return contentDigests.get(algo);
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static ByteBuffer sliceFromTo(ByteBuffer source, int start, int end) {
        if (start < 0) {
            throw new IllegalArgumentException("start: " + start);
        }
        if (end < start) {
            throw new IllegalArgumentException("end < start: " + end + " < " + start);
        }
        int capacity = source.capacity();
        if (end > source.capacity()) {
            throw new IllegalArgumentException("end > capacity: " + end + " > " + capacity);
        }
        int originalLimit = source.limit();
        int originalPosition = source.position();
        try {
            source.position(0);
            source.limit(end);
            source.position(start);
            ByteBuffer result = source.slice();
            result.order(source.order());
            ByteBuffer byteBuffer = result;
            return byteBuffer;
        }
        finally {
            source.position(0);
            source.limit(originalLimit);
            source.position(originalPosition);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static ByteBuffer getByteBuffer(ByteBuffer source, int size) throws BufferUnderflowException {
        if (size < 0) {
            throw new IllegalArgumentException("size: " + size);
        }
        int originalLimit = source.limit();
        int position = source.position();
        int limit = position + size;
        if (limit < position || limit > originalLimit) {
            throw new BufferUnderflowException();
        }
        source.limit(limit);
        try {
            ByteBuffer result = source.slice();
            result.order(source.order());
            source.position(limit);
            ByteBuffer byteBuffer = result;
            return byteBuffer;
        }
        finally {
            source.limit(originalLimit);
        }
    }

    static ByteBuffer getLengthPrefixedSlice(ByteBuffer source) throws IOException {
        if (source.remaining() < 4) {
            throw new IOException("Remaining buffer too short to contain length of length-prefixed field. Remaining: " + source.remaining());
        }
        int len = source.getInt();
        if (len < 0) {
            throw new IllegalArgumentException("Negative length");
        }
        if (len > source.remaining()) {
            throw new IOException("Length-prefixed field longer than remaining buffer. Field length: " + len + ", remaining: " + source.remaining());
        }
        return ApkSigningBlockUtils.getByteBuffer(source, len);
    }

    static byte[] readLengthPrefixedByteArray(ByteBuffer buf) throws IOException {
        int len = buf.getInt();
        if (len < 0) {
            throw new IOException("Negative length");
        }
        if (len > buf.remaining()) {
            throw new IOException("Underflow while reading length-prefixed value. Length: " + len + ", available: " + buf.remaining());
        }
        byte[] result = new byte[len];
        buf.get(result);
        return result;
    }

    static void setUnsignedInt32LittleEndian(int value, byte[] result, int offset) {
        result[offset] = (byte)(value & 0xFF);
        result[offset + 1] = (byte)(value >>> 8 & 0xFF);
        result[offset + 2] = (byte)(value >>> 16 & 0xFF);
        result[offset + 3] = (byte)(value >>> 24 & 0xFF);
    }

    static Pair<ByteBuffer, Long> findApkSigningBlock(RandomAccessFile apk, long centralDirOffset) throws IOException, SignatureNotFoundException {
        if (centralDirOffset < 32L) {
            throw new SignatureNotFoundException("APK too small for APK Signing Block. ZIP Central Directory offset: " + centralDirOffset);
        }
        ByteBuffer footer = ByteBuffer.allocate(24);
        footer.order(ByteOrder.LITTLE_ENDIAN);
        apk.seek(centralDirOffset - (long)footer.capacity());
        apk.readFully(footer.array(), footer.arrayOffset(), footer.capacity());
        if (footer.getLong(8) != 2334950737559900225L || footer.getLong(16) != 3617552046287187010L) {
            throw new SignatureNotFoundException("No APK Signing Block before ZIP Central Directory");
        }
        long apkSigBlockSizeInFooter = footer.getLong(0);
        if (apkSigBlockSizeInFooter < (long)footer.capacity() || apkSigBlockSizeInFooter > 0x7FFFFFF7L) {
            throw new SignatureNotFoundException("APK Signing Block size out of range: " + apkSigBlockSizeInFooter);
        }
        int totalSize = (int)(apkSigBlockSizeInFooter + 8L);
        long apkSigBlockOffset = centralDirOffset - (long)totalSize;
        if (apkSigBlockOffset < 0L) {
            throw new SignatureNotFoundException("APK Signing Block offset out of range: " + apkSigBlockOffset);
        }
        ByteBuffer apkSigBlock = ByteBuffer.allocate(totalSize);
        apkSigBlock.order(ByteOrder.LITTLE_ENDIAN);
        apk.seek(apkSigBlockOffset);
        apk.readFully(apkSigBlock.array(), apkSigBlock.arrayOffset(), apkSigBlock.capacity());
        long apkSigBlockSizeInHeader = apkSigBlock.getLong(0);
        if (apkSigBlockSizeInHeader != apkSigBlockSizeInFooter) {
            throw new SignatureNotFoundException("APK Signing Block sizes in header and footer do not match: " + apkSigBlockSizeInHeader + " vs " + apkSigBlockSizeInFooter);
        }
        return Pair.create(apkSigBlock, apkSigBlockOffset);
    }

    static ByteBuffer findApkSignatureSchemeBlock(ByteBuffer apkSigningBlock, int blockId) throws SignatureNotFoundException {
        ApkSigningBlockUtils.checkByteOrderLittleEndian(apkSigningBlock);
        ByteBuffer pairs = ApkSigningBlockUtils.sliceFromTo(apkSigningBlock, 8, apkSigningBlock.capacity() - 24);
        int entryCount = 0;
        while (pairs.hasRemaining()) {
            ++entryCount;
            if (pairs.remaining() < 8) {
                throw new SignatureNotFoundException("Insufficient data to read size of APK Signing Block entry #" + entryCount);
            }
            long lenLong = pairs.getLong();
            if (lenLong < 4L || lenLong > Integer.MAX_VALUE) {
                throw new SignatureNotFoundException("APK Signing Block entry #" + entryCount + " size out of range: " + lenLong);
            }
            int len = (int)lenLong;
            int nextEntryPos = pairs.position() + len;
            if (len > pairs.remaining()) {
                throw new SignatureNotFoundException("APK Signing Block entry #" + entryCount + " size out of range: " + len + ", available: " + pairs.remaining());
            }
            int id2 = pairs.getInt();
            if (id2 == blockId) {
                return ApkSigningBlockUtils.getByteBuffer(pairs, len - 4);
            }
            pairs.position(nextEntryPos);
        }
        throw new SignatureNotFoundException("No block with ID " + blockId + " in APK Signing Block.");
    }

    private static void checkByteOrderLittleEndian(ByteBuffer buffer) {
        if (buffer.order() != ByteOrder.LITTLE_ENDIAN) {
            throw new IllegalArgumentException("ByteBuffer byte order must be little endian");
        }
    }

    private static class MultipleDigestDataDigester
    implements DataDigester {
        private final MessageDigest[] mMds;

        MultipleDigestDataDigester(MessageDigest[] mds) {
            this.mMds = mds;
        }

        @Override
        public void consume(ByteBuffer buffer) {
            buffer = buffer.slice();
            for (MessageDigest md : this.mMds) {
                buffer.position(0);
                md.update(buffer);
            }
        }
    }
}

