/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.codeEditor.printing;

import com.intellij.application.options.CodeStyle;
import com.intellij.codeEditor.printing.ExportToHTMLManager;
import com.intellij.codeEditor.printing.FileSeparatorProvider;
import com.intellij.codeEditor.printing.HtmlStyleManager;
import com.intellij.codeInsight.daemon.LineMarkerInfo;
import com.intellij.ide.highlighter.HighlighterFactory;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.colors.EditorColorsManager;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.highlighter.EditorHighlighter;
import com.intellij.openapi.editor.highlighter.HighlighterIterator;
import com.intellij.openapi.editor.markup.TextAttributes;
import com.intellij.openapi.fileTypes.FileType;
import com.intellij.openapi.progress.ProcessCanceledException;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.util.Comparing;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.psi.PsiDocumentManager;
import com.intellij.psi.PsiElement;
import com.intellij.psi.PsiFile;
import com.intellij.psi.PsiReference;
import com.intellij.psi.impl.file.PsiDirectoryFactory;
import com.intellij.ui.ColorUtil;
import com.intellij.ui.Gray;
import com.intellij.ui.JBColor;
import it.unimi.dsi.fastutil.ints.Int2ObjectRBTreeMap;
import it.unimi.dsi.fastutil.ints.IntBidirectionalIterator;
import java.awt.Color;
import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.StringTokenizer;
import org.jetbrains.annotations.NonNls;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class HTMLTextPainter {
    private static final Logger LOG = Logger.getInstance(HTMLTextPainter.class);
    private int myOffset;
    private final EditorHighlighter myHighlighter;
    private final String myText;
    private final String myFileName;
    private int mySegmentEnd;
    private final PsiFile myPsiFile;
    private final Document myDocument;
    private int lineCount;
    private int myFirstLineNumber;
    private final boolean myPrintLineNumbers;
    private int myColumn;
    private final List<LineMarkerInfo<?>> myMethodSeparators = new ArrayList();
    private int myCurrentMethodSeparator;
    private final Project myProject;
    private final HtmlStyleManager htmlStyleManager;

    HTMLTextPainter(@NotNull PsiFile psiFile, @NotNull Project project, boolean printLineNumbers) {
        this(psiFile, project, new HtmlStyleManager(false), printLineNumbers, true);
    }

    public HTMLTextPainter(@NotNull PsiFile psiFile, @NotNull Project project, @NotNull HtmlStyleManager htmlStyleManager, boolean printLineNumbers, boolean useMethodSeparators) {
        this.myProject = project;
        this.myPsiFile = psiFile;
        this.htmlStyleManager = htmlStyleManager;
        this.myPrintLineNumbers = printLineNumbers;
        this.myHighlighter = HighlighterFactory.createHighlighter((Project)project, (VirtualFile)psiFile.getVirtualFile());
        this.myText = psiFile.getText();
        this.myHighlighter.setText((CharSequence)this.myText);
        this.mySegmentEnd = this.myText.length();
        this.myFileName = psiFile.getVirtualFile().getPresentableUrl();
        this.myDocument = PsiDocumentManager.getInstance((Project)project).getDocument(psiFile);
        if (useMethodSeparators && this.myDocument != null) {
            this.myMethodSeparators.addAll(FileSeparatorProvider.getFileSeparators(psiFile, this.myDocument));
        }
        this.myCurrentMethodSeparator = 0;
    }

    private HTMLTextPainter(@NotNull PsiElement context, @NotNull String codeFragment) {
        this.myProject = context.getProject();
        this.myPsiFile = context.getContainingFile();
        if (this.myPsiFile == null) {
            throw new IllegalArgumentException("Bad context: no container file");
        }
        this.htmlStyleManager = new HtmlStyleManager(true);
        this.myPrintLineNumbers = false;
        this.myHighlighter = HighlighterFactory.createHighlighter((Project)this.myProject, (FileType)this.myPsiFile.getFileType());
        this.myText = codeFragment;
        this.myHighlighter.setText((CharSequence)this.myText);
        this.mySegmentEnd = this.myText.length();
        this.myFileName = "fragment";
        this.myDocument = null;
        this.myCurrentMethodSeparator = 0;
    }

    @NotNull
    public PsiFile getPsiFile() {
        return this.myPsiFile;
    }

    void setSegment(int segmentStart, int segmentEnd, int firstLineNumber) {
        this.myOffset = segmentStart;
        this.mySegmentEnd = segmentEnd;
        this.myFirstLineNumber = firstLineNumber;
    }

    public void paint(@Nullable Int2ObjectRBTreeMap<PsiReference> refMap, @NotNull Writer writer, boolean isStandalone) throws IOException {
        HighlighterIterator hIterator = this.myHighlighter.createIterator(this.myOffset);
        if (hIterator.atEnd()) {
            return;
        }
        this.lineCount = this.myFirstLineNumber;
        TextAttributes prevAttributes = null;
        IntBidirectionalIterator refKeys = null;
        int refOffset = -1;
        PsiReference ref = null;
        if (refMap != null && (refKeys = refMap.keySet().iterator()).hasNext()) {
            int key = refKeys.nextInt();
            ref = (PsiReference)refMap.get(key);
            refOffset = key;
        }
        int referenceEnd = -1;
        if (isStandalone) {
            this.writeHeader(writer, new File(this.myFileName).getName());
        } else {
            this.ensureStyles();
        }
        writer.write("<pre>");
        if (this.myFirstLineNumber == 0) {
            this.writeLineNumber(writer);
        }
        String closeTag = null;
        this.getMethodSeparator(hIterator.getStart());
        while (!hIterator.atEnd()) {
            TextAttributes textAttributes;
            char c;
            int hStart;
            int hEnd = hIterator.getEnd();
            if (hEnd > this.mySegmentEnd) break;
            for (hStart = hIterator.getStart(); hStart < hEnd && Character.isWhitespace(c = this.myText.charAt(hStart)); ++hStart) {
                if (closeTag != null && c == '\n') {
                    writer.write(closeTag);
                    closeTag = null;
                    prevAttributes = null;
                }
                if (c == '\n') {
                    this.writeLineSeparatorAndNumber(writer, hStart);
                    continue;
                }
                writer.write(c);
            }
            if (hStart == hEnd) {
                hIterator.advance();
                continue;
            }
            if (refOffset > 0 && hStart <= refOffset && hEnd > refOffset) {
                referenceEnd = this.writeReferenceTag(writer, ref);
            }
            if (this.htmlStyleManager.isDefaultAttributes(textAttributes = hIterator.getTextAttributes())) {
                textAttributes = null;
            }
            if (!HTMLTextPainter.equals(prevAttributes, textAttributes) && referenceEnd < 0) {
                if (closeTag != null) {
                    writer.write(closeTag);
                    closeTag = null;
                }
                if (textAttributes != null) {
                    this.htmlStyleManager.writeTextStyle(writer, textAttributes);
                    closeTag = "</span>";
                }
                prevAttributes = textAttributes;
            }
            this.writeString(writer, this.myText, hStart, hEnd - hStart, this.myPsiFile);
            if (referenceEnd > 0 && hEnd >= referenceEnd) {
                writer.write("</a>");
                referenceEnd = -1;
                if (refKeys.hasNext()) {
                    int key = refKeys.nextInt();
                    ref = (PsiReference)refMap.get(key);
                    refOffset = key;
                }
            }
            hIterator.advance();
        }
        if (closeTag != null) {
            writer.write(closeTag);
        }
        writer.write("</pre>\n");
        if (isStandalone) {
            writer.write("</body>\n");
            writer.write("</html>");
        }
    }

    private void ensureStyles() {
        this.htmlStyleManager.ensureStyles(this.myHighlighter.createIterator(this.myOffset), this.myMethodSeparators);
    }

    @Nullable
    private LineMarkerInfo<?> getMethodSeparator(int offset) {
        LineMarkerInfo<?> tmpMarker;
        if (this.myDocument == null) {
            return null;
        }
        int line = this.myDocument.getLineNumber(Math.max(0, Math.min(this.myDocument.getTextLength(), offset)));
        LineMarkerInfo<?> marker = null;
        while (this.myCurrentMethodSeparator < this.myMethodSeparators.size() && (tmpMarker = this.myMethodSeparators.get(this.myCurrentMethodSeparator)) != null && FileSeparatorProvider.getDisplayLine(tmpMarker, this.myDocument) <= line) {
            marker = tmpMarker;
            ++this.myCurrentMethodSeparator;
        }
        return marker;
    }

    private int writeReferenceTag(Writer writer, PsiReference ref) throws IOException {
        PsiFile refFile = Objects.requireNonNull(ref.resolve()).getContainingFile();
        PsiDirectoryFactory psiDirectoryFactory = PsiDirectoryFactory.getInstance(this.myProject);
        String refPackageName = psiDirectoryFactory.getQualifiedName(refFile.getContainingDirectory(), false);
        String psiPackageName = psiDirectoryFactory.getQualifiedName(this.myPsiFile.getContainingDirectory(), false);
        StringBuilder fileName = new StringBuilder();
        if (!psiPackageName.equals(refPackageName)) {
            StringTokenizer tokens = new StringTokenizer(psiPackageName, ".");
            while (tokens.hasMoreTokens()) {
                tokens.nextToken();
                fileName.append("../");
            }
            StringTokenizer refTokens = new StringTokenizer(refPackageName, ".");
            while (refTokens.hasMoreTokens()) {
                String token = refTokens.nextToken();
                fileName.append(token);
                fileName.append('/');
            }
        }
        fileName.append(ExportToHTMLManager.getHTMLFileName(refFile));
        writer.write("<a href=\"" + fileName + "\">");
        return ref.getElement().getTextRange().getEndOffset();
    }

    private void writeString(Writer writer, CharSequence charArray, int start2, int length, @NotNull PsiFile psiFile) throws IOException {
        for (int i2 = start2; i2 < start2 + length; ++i2) {
            char c = charArray.charAt(i2);
            if (c == '<') {
                this.writeChar(writer, "&lt;");
                continue;
            }
            if (c == '>') {
                this.writeChar(writer, "&gt;");
                continue;
            }
            if (c == '&') {
                this.writeChar(writer, "&amp;");
                continue;
            }
            if (c == '\"') {
                this.writeChar(writer, "&quot;");
                continue;
            }
            if (c == '\t') {
                int tabSize = CodeStyle.getIndentOptions((PsiFile)psiFile).TAB_SIZE;
                if (tabSize <= 0) {
                    tabSize = 1;
                }
                int nSpaces = tabSize - this.myColumn % tabSize;
                for (int j = 0; j < nSpaces; ++j) {
                    this.writeChar(writer, " ");
                }
                continue;
            }
            if (c == '\n' || c == '\r') {
                if (c == '\r' && i2 + 1 < start2 + length && charArray.charAt(i2 + 1) == '\n') {
                    ++i2;
                } else if (c == '\n') {
                    this.writeChar(writer, " ");
                }
                this.writeLineSeparatorAndNumber(writer, i2);
                continue;
            }
            writer.write(c);
            ++this.myColumn;
        }
    }

    private void writeLineSeparatorAndNumber(@NotNull Writer writer, int i2) throws IOException {
        LineMarkerInfo<?> marker = this.getMethodSeparator(i2 + 1);
        if (marker == null) {
            writer.write(10);
        } else {
            writer.write("<hr class=\"" + this.htmlStyleManager.getSeparatorClassName(marker.separatorColor) + "\">");
        }
        this.writeLineNumber(writer);
    }

    private void writeChar(Writer writer, String s) throws IOException {
        writer.write(s);
        ++this.myColumn;
    }

    private void writeLineNumber(@NonNls Writer writer) throws IOException {
        this.myColumn = 0;
        ++this.lineCount;
        if (this.myPrintLineNumbers) {
            writer.write("<a name=\"l" + this.lineCount + "\">");
            writer.write("<span class=\"ln\">");
            String s = Integer.toString(this.lineCount);
            writer.write(s);
            int extraSpaces = 4 - s.length();
            do {
                writer.write(32);
            } while (extraSpaces-- > 0);
            writer.write("</span></a>");
        }
    }

    private void writeHeader(@NonNls Writer writer, @Nullable String title) throws IOException {
        writer.write("<html>\n");
        writer.write("<head>\n");
        writer.write("<title>" + title + "</title>\n");
        writer.write("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=utf-8\">\n");
        this.ensureStyles();
        this.htmlStyleManager.writeStyleTag(writer, this.myPrintLineNumbers);
        writer.write("</head>\n");
        EditorColorsScheme scheme = EditorColorsManager.getInstance().getGlobalScheme();
        writer.write("<body bgcolor=\"" + ColorUtil.toHtmlColor((Color)scheme.getDefaultBackground()) + "\">\n");
        writer.write("<table CELLSPACING=0 CELLPADDING=5 COLS=1 WIDTH=\"100%\" BGCOLOR=\"#" + ColorUtil.toHex((Color)new JBColor((Color)Gray.xC0, (Color)Gray.x60)) + "\" >\n");
        writer.write("<tr><td><center>\n");
        writer.write("<font face=\"Arial, Helvetica\" color=\"#000000\">\n");
        writer.write(title + "</font>\n");
        writer.write("</center></td></tr></table>\n");
    }

    private static boolean equals(TextAttributes attributes1, TextAttributes attributes2) {
        if (attributes2 == null) {
            return attributes1 == null;
        }
        if (attributes1 == null) {
            return false;
        }
        if (!Comparing.equal((Object)attributes1.getForegroundColor(), (Object)attributes2.getForegroundColor())) {
            return false;
        }
        if (attributes1.getFontType() != attributes2.getFontType()) {
            return false;
        }
        if (!Comparing.equal((Object)attributes1.getBackgroundColor(), (Object)attributes2.getBackgroundColor())) {
            return false;
        }
        return Comparing.equal((Object)attributes1.getEffectColor(), (Object)attributes2.getEffectColor());
    }

    @NotNull
    public static String convertCodeFragmentToHTMLFragmentWithInlineStyles(@NotNull PsiElement context, @NotNull String codeFragment) {
        try {
            StringWriter writer = new StringWriter();
            new HTMLTextPainter(context, codeFragment).paint(null, writer, false);
            return writer.toString();
        }
        catch (ProcessCanceledException cancel) {
            throw cancel;
        }
        catch (Throwable e) {
            LOG.error(e);
            return String.format("<pre>%s</pre>\n", codeFragment);
        }
    }
}

