from abc import ABC, abstractmethod
from typing import List, Optional

import torch
from PIL import Image


class AbstractMultimodalPipeline(ABC):
    @staticmethod
    @abstractmethod
    def name() -> str:
        'имя pipeline должно быть таким же, как в --multimodal-pipeline'
        pass

    @staticmethod
    @abstractmethod
    def image_start() -> Optional[str]:
        'вернуть начальную строку изображения, строковое представление начального токена изображения или None, если это неприменимо'
        pass

    @staticmethod
    @abstractmethod
    def image_end() -> Optional[str]:
        'вернуть строку конца изображения, строковое представление токена конца изображения или None, если это неприменимо'
        pass

    @staticmethod
    @abstractmethod
    def placeholder_token_id() -> int:
        'вернуть идентификатор токена-заполнителя'
        pass

    @staticmethod
    @abstractmethod
    def num_image_embeds() -> int:
        'вернуть количество внедрений, использованных одним изображением (например: 256 для LLaVA)'
        pass

    @abstractmethod
    def embed_images(self, images: List[Image.Image]) -> torch.Tensor:
        'пересылать изображения через pipeline видения и возвращать их вложения'
        pass

    @staticmethod
    @abstractmethod
    def embed_tokens(input_ids: torch.Tensor) -> torch.Tensor:
        'встроенные токены, точная функция зависит от LLM, для LLaMA это `shared.model.model.embed_tokens`'
        pass

    @staticmethod
    @abstractmethod
    def placeholder_embeddings() -> torch.Tensor:
        'получить вложения заполнителей, если есть несколько изображений, и `add_all_images_to_prompt` неверно'
        pass

    def _get_device(self, setting_name: str, params: dict):
        if params[setting_name] is None:
            return torch.device("cuda:0" if torch.cuda.is_available() else "cpu")
        return torch.device(params[setting_name])

    def _get_dtype(self, setting_name: str, params: dict):
        return torch.float32 if int(params[setting_name]) == 32 else torch.float16
