/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.cryptofs;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystemAlreadyExistsException;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.security.SecureRandom;
import java.util.EnumSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.cryptomator.cryptofs.ContentRootMissingException;
import org.cryptomator.cryptofs.CryptoFileSystemComponent;
import org.cryptomator.cryptofs.CryptoFileSystemImpl;
import org.cryptomator.cryptofs.CryptoFileSystemProperties;
import org.cryptomator.cryptofs.CryptoFileSystemProvider;
import org.cryptomator.cryptofs.FileSystemNeedsMigrationException;
import org.cryptomator.cryptofs.VaultConfig;
import org.cryptomator.cryptofs.common.BackupHelper;
import org.cryptomator.cryptofs.common.FileSystemCapabilityChecker;
import org.cryptomator.cryptolib.api.Cryptor;
import org.cryptomator.cryptolib.api.CryptorProvider;
import org.cryptomator.cryptolib.api.Masterkey;
import org.cryptomator.cryptolib.api.MasterkeyLoadingFailedException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
class CryptoFileSystems {
    private static final Logger LOG = LoggerFactory.getLogger(CryptoFileSystems.class);
    private final ConcurrentMap<Path, CryptoFileSystemImpl> fileSystems = new ConcurrentHashMap<Path, CryptoFileSystemImpl>();
    private final CryptoFileSystemComponent.Factory cryptoFileSystemComponentFactory;
    private final FileSystemCapabilityChecker capabilityChecker;
    private final SecureRandom csprng;

    @Inject
    public CryptoFileSystems(CryptoFileSystemComponent.Factory cryptoFileSystemComponentFactory, FileSystemCapabilityChecker capabilityChecker, SecureRandom csprng) {
        this.cryptoFileSystemComponentFactory = cryptoFileSystemComponentFactory;
        this.capabilityChecker = capabilityChecker;
        this.csprng = csprng;
    }

    public CryptoFileSystemImpl create(CryptoFileSystemProvider provider, Path pathToVault, CryptoFileSystemProperties properties) throws IOException, MasterkeyLoadingFailedException {
        Path normalizedPathToVault = pathToVault.normalize();
        String token = this.readVaultConfigFile(normalizedPathToVault, properties);
        VaultConfig.UnverifiedVaultConfig configLoader = VaultConfig.decode(token);
        URI keyId = configLoader.getKeyId();
        try (Masterkey key = properties.keyLoader().loadKey(keyId);){
            VaultConfig config = configLoader.verify(key.getEncoded(), 8);
            this.backupVaultConfigFile(normalizedPathToVault, properties);
            CryptoFileSystemProperties adjustedProperties = this.adjustForCapabilities(pathToVault, properties);
            Cryptor cryptor = CryptorProvider.forScheme((CryptorProvider.Scheme)config.getCipherCombo()).provide(key.copy(), this.csprng);
            try {
                this.checkVaultRootExistence(pathToVault, cryptor);
                CryptoFileSystemImpl cryptoFileSystemImpl = this.fileSystems.compute(normalizedPathToVault, (path, fs) -> {
                    if (fs == null) {
                        return this.cryptoFileSystemComponentFactory.create(cryptor, config, provider, normalizedPathToVault, adjustedProperties).cryptoFileSystem();
                    }
                    throw new FileSystemAlreadyExistsException();
                });
                return cryptoFileSystemImpl;
            }
            catch (Exception e) {
                cryptor.destroy();
                throw e;
            }
        }
    }

    private void checkVaultRootExistence(Path pathToVault, Cryptor cryptor) throws ContentRootMissingException {
        String dirHash = cryptor.fileNameCryptor().hashDirectoryId("");
        Path vaultCipherRootPath = pathToVault.resolve("d").resolve(dirHash.substring(0, 2)).resolve(dirHash.substring(2));
        if (!Files.exists(vaultCipherRootPath, new LinkOption[0])) {
            throw new ContentRootMissingException("The encrypted root directory of the vault " + String.valueOf(pathToVault) + " is missing.");
        }
    }

    private String readVaultConfigFile(Path pathToVault, CryptoFileSystemProperties properties) throws IOException, FileSystemNeedsMigrationException {
        Path vaultConfigFile = pathToVault.resolve(properties.vaultConfigFilename());
        try {
            return Files.readString(vaultConfigFile, StandardCharsets.US_ASCII);
        }
        catch (NoSuchFileException e) {
            String masterkeyFilename = properties.masterkeyFilename();
            if (masterkeyFilename != null && Files.exists(pathToVault.resolve(masterkeyFilename), new LinkOption[0])) {
                LOG.warn("Failed to read {}, but found {}}", (Object)vaultConfigFile, (Object)masterkeyFilename);
                throw new FileSystemNeedsMigrationException(pathToVault);
            }
            throw e;
        }
    }

    private void backupVaultConfigFile(Path pathToVault, CryptoFileSystemProperties properties) throws IOException {
        Path vaultConfigFile = pathToVault.resolve(properties.vaultConfigFilename());
        BackupHelper.attemptBackup(vaultConfigFile);
    }

    private CryptoFileSystemProperties adjustForCapabilities(Path pathToVault, CryptoFileSystemProperties originalProperties) throws FileSystemCapabilityChecker.MissingCapabilityException {
        if (!originalProperties.readonly()) {
            try {
                this.capabilityChecker.assertWriteAccess(pathToVault);
                return originalProperties;
            }
            catch (FileSystemCapabilityChecker.MissingCapabilityException e) {
                this.capabilityChecker.assertReadAccess(pathToVault);
                LOG.warn("No write access to vault. Fallback to read-only access.");
                EnumSet<CryptoFileSystemProperties.FileSystemFlags> flags = EnumSet.copyOf(originalProperties.flags());
                flags.add(CryptoFileSystemProperties.FileSystemFlags.READONLY);
                return CryptoFileSystemProperties.cryptoFileSystemPropertiesFrom(originalProperties).withFlags(flags).build();
            }
        }
        return originalProperties;
    }

    public void remove(CryptoFileSystemImpl cryptoFileSystem) {
        this.fileSystems.values().remove(cryptoFileSystem);
    }

    public boolean contains(CryptoFileSystemImpl cryptoFileSystem) {
        return this.fileSystems.containsValue(cryptoFileSystem);
    }

    public CryptoFileSystemImpl get(Path pathToVault) {
        Path normalizedPathToVault = pathToVault.normalize();
        CryptoFileSystemImpl fs = (CryptoFileSystemImpl)this.fileSystems.get(normalizedPathToVault);
        if (fs == null) {
            throw new FileSystemNotFoundException(String.format("CryptoFileSystem at %s not initialized", normalizedPathToVault));
        }
        return fs;
    }
}

