/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.common.settings;

import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import java.io.IOException;
import java.nio.CharBuffer;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.security.SecureRandom;
import java.util.Arrays;
import java.util.UUID;
import java.util.function.Supplier;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.cryptomator.common.Environment;
import org.cryptomator.common.keychain.KeychainManager;
import org.cryptomator.cryptolib.common.P384KeyPair;
import org.cryptomator.cryptolib.common.Pkcs12Exception;
import org.cryptomator.integrations.keychain.KeychainAccessException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class DeviceKey {
    private static final Logger LOG = LoggerFactory.getLogger(DeviceKey.class);
    private static final String KEYCHAIN_KEY = "cryptomator-device-p12";
    private static final String KEYCHAIN_DISPLAY_NAME = "Cryptomator Device Keypair .p12 Passphrase";
    private final KeychainManager keychainManager;
    private final Environment env;
    private final SecureRandom csprng;
    private final Supplier<P384KeyPair> keyPairSupplier;

    @Inject
    public DeviceKey(KeychainManager keychainManager, Environment env, SecureRandom csprng) {
        this.keychainManager = keychainManager;
        this.env = env;
        this.csprng = csprng;
        this.keyPairSupplier = Suppliers.memoize(this::loadOrCreate);
    }

    public P384KeyPair get() throws DeviceKeyRetrievalException {
        Preconditions.checkState((boolean)this.keychainManager.isSupported());
        return this.keyPairSupplier.get();
    }

    private P384KeyPair loadOrCreate() throws DeviceKeyRetrievalException {
        Path p12File = this.env.getP12Path().findFirst().orElseThrow(() -> new DeviceKeyRetrievalException("No path for .p12 file configured"));
        char[] passphrase = null;
        try {
            passphrase = this.keychainManager.loadPassphrase(KEYCHAIN_KEY);
            if (passphrase != null && Files.isRegularFile(p12File, new LinkOption[0])) {
                P384KeyPair p384KeyPair = this.loadExistingKeyPair(passphrase, p12File);
                return p384KeyPair;
            }
            passphrase = this.randomPassword();
            this.keychainManager.storePassphrase(KEYCHAIN_KEY, KEYCHAIN_DISPLAY_NAME, CharBuffer.wrap(passphrase));
            P384KeyPair p384KeyPair = this.createAndStoreNewKeyPair(passphrase, p12File);
            return p384KeyPair;
        }
        catch (KeychainAccessException e) {
            throw new DeviceKeyRetrievalException("Failed to access system keychain", e);
        }
        catch (IOException | Pkcs12Exception e) {
            throw new DeviceKeyRetrievalException("Failed to access .p12 file", e);
        }
        finally {
            if (passphrase != null) {
                Arrays.fill(passphrase, '\u0000');
            }
        }
    }

    private P384KeyPair loadExistingKeyPair(char[] passphrase, Path p12File) throws IOException {
        LOG.debug("Loading existing device key from {}", (Object)p12File);
        return P384KeyPair.load((Path)p12File, (char[])passphrase);
    }

    private P384KeyPair createAndStoreNewKeyPair(char[] passphrase, Path p12File) throws IOException {
        P384KeyPair keyPair = P384KeyPair.generate();
        Path tmpFile = p12File.resolveSibling(p12File.getFileName().toString() + ".tmp");
        if (Files.exists(tmpFile, new LinkOption[0])) {
            LOG.debug("Leftover from devicekey creation detected. Deleting {}", (Object)tmpFile);
            Files.delete(tmpFile);
        }
        LOG.debug("Store new device key to {}", (Object)p12File);
        keyPair.store(p12File, passphrase);
        return keyPair;
    }

    private char[] randomPassword() {
        UUID uuid = new UUID(this.csprng.nextLong(), this.csprng.nextLong());
        return uuid.toString().toCharArray();
    }

    public static class DeviceKeyRetrievalException
    extends RuntimeException {
        private DeviceKeyRetrievalException(String message) {
            super(message);
        }

        private DeviceKeyRetrievalException(String message, Throwable cause) {
            super(message, cause);
        }
    }
}

