/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.common.mount;

import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.stream.Stream;
import org.apache.commons.lang3.SystemUtils;
import org.cryptomator.common.mount.HideawayNotDirectoryException;
import org.cryptomator.common.mount.IllegalMountPointException;
import org.cryptomator.common.mount.MountPointCleanupFailedException;
import org.cryptomator.common.mount.MountPointInUseException;
import org.cryptomator.common.mount.MountPointNotEmptyDirectoryException;
import org.cryptomator.common.mount.MountPointNotExistingException;
import org.cryptomator.common.mount.Mounter;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class MountWithinParentUtil {
    private static final Logger LOG = LoggerFactory.getLogger(Mounter.class);
    private static final String HIDEAWAY_PREFIX = ".~$";
    private static final String HIDEAWAY_SUFFIX = ".tmp";
    private static final String WIN_HIDDEN_ATTR = "dos:hidden";

    private MountWithinParentUtil() {
    }

    static void prepareParentNoMountPoint(Path mountPoint) throws IllegalMountPointException, IOException {
        Path hideaway = MountWithinParentUtil.getHideaway(mountPoint);
        MountPointState mpState = MountWithinParentUtil.getMountPointState(mountPoint);
        boolean hideExists = Files.exists(hideaway, LinkOption.NOFOLLOW_LINKS);
        if (mpState == MountPointState.BROKEN_JUNCTION) {
            LOG.info("Mountpoint \"{}\" is still a junction. Deleting it.", (Object)mountPoint);
            Files.delete(mountPoint);
            mpState = MountPointState.NOT_EXISTING;
        }
        if (mpState == MountPointState.NOT_EXISTING && !hideExists) {
            throw new MountPointNotExistingException(mountPoint);
        }
        if (mpState == MountPointState.NOT_EXISTING) {
            MountWithinParentUtil.checkIsHideawayDirectory(mountPoint, hideaway);
            LOG.info("Mountpoint {} seems to be not properly cleaned up. Will be fixed on unmount.", (Object)mountPoint);
            if (SystemUtils.IS_OS_WINDOWS) {
                Files.setAttribute(hideaway, WIN_HIDDEN_ATTR, true, LinkOption.NOFOLLOW_LINKS);
            }
        } else {
            assert (mpState == MountPointState.EMPTY_DIR);
            try {
                if (hideExists) {
                    MountWithinParentUtil.removeResidualHideaway(mountPoint, hideaway);
                }
                Files.move(mountPoint, hideaway, new CopyOption[0]);
                if (SystemUtils.IS_OS_WINDOWS) {
                    Files.setAttribute(hideaway, WIN_HIDDEN_ATTR, true, LinkOption.NOFOLLOW_LINKS);
                }
                int attempts = 0;
                while (!Files.notExists(mountPoint, new LinkOption[0])) {
                    if (attempts >= 10) {
                        throw new MountPointCleanupFailedException(mountPoint);
                    }
                    Thread.sleep(1000L);
                    ++attempts;
                }
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new RuntimeException(e);
            }
        }
    }

    @VisibleForTesting
    static MountPointState getMountPointState(Path path) throws IOException, IllegalMountPointException {
        if (Files.notExists(path, LinkOption.NOFOLLOW_LINKS)) {
            return MountPointState.NOT_EXISTING;
        }
        if (!Files.readAttributes(path, BasicFileAttributes.class, LinkOption.NOFOLLOW_LINKS).isOther()) {
            MountWithinParentUtil.checkIsMountPointDirectory(path);
            MountWithinParentUtil.checkIsMountPointEmpty(path);
            return MountPointState.EMPTY_DIR;
        }
        if (Files.exists(path, new LinkOption[0])) {
            throw new MountPointInUseException(path);
        }
        return MountPointState.BROKEN_JUNCTION;
    }

    @VisibleForTesting
    static void removeResidualHideaway(Path mountPoint, Path hideaway) throws IOException {
        MountWithinParentUtil.checkIsHideawayDirectory(mountPoint, hideaway);
        Files.delete(hideaway);
    }

    static void cleanup(Path mountPoint) {
        Path hideaway = MountWithinParentUtil.getHideaway(mountPoint);
        try {
            MountWithinParentUtil.waitForMountpointRestoration(mountPoint);
            if (Files.notExists(hideaway, LinkOption.NOFOLLOW_LINKS)) {
                LOG.error("Unable to restore hidden directory to mountpoint \"{}\": Directory does not exist.", (Object)mountPoint);
                return;
            }
            Files.move(hideaway, mountPoint, new CopyOption[0]);
            if (SystemUtils.IS_OS_WINDOWS) {
                Files.setAttribute(mountPoint, WIN_HIDDEN_ATTR, false, new LinkOption[0]);
            }
        }
        catch (IOException e) {
            LOG.error("Unable to restore hidden directory to mountpoint \"{}\".", (Object)mountPoint, (Object)e);
        }
    }

    private static void waitForMountpointRestoration(Path mountPoint) throws FileAlreadyExistsException {
        int attempts = 0;
        while (!Files.notExists(mountPoint, LinkOption.NOFOLLOW_LINKS)) {
            if (++attempts >= 5) {
                throw new FileAlreadyExistsException("Timeout waiting for mountpoint cleanup for " + String.valueOf(mountPoint) + " .");
            }
            try {
                Thread.sleep(300L);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw new FileAlreadyExistsException("Interrupted before mountpoint " + String.valueOf(mountPoint) + " was cleared");
            }
        }
    }

    private static void checkIsMountPointDirectory(Path toCheck) throws IllegalMountPointException {
        if (!Files.isDirectory(toCheck, LinkOption.NOFOLLOW_LINKS)) {
            throw new MountPointNotEmptyDirectoryException(toCheck, "Mountpoint is not a directory: " + String.valueOf(toCheck));
        }
    }

    private static void checkIsHideawayDirectory(Path mountPoint, Path hideawayToCheck) {
        if (!Files.isDirectory(hideawayToCheck, LinkOption.NOFOLLOW_LINKS)) {
            throw new HideawayNotDirectoryException(mountPoint, hideawayToCheck);
        }
    }

    private static void checkIsMountPointEmpty(Path toCheck) throws IllegalMountPointException, IOException {
        try (Stream<Path> dirStream = Files.list(toCheck);){
            if (dirStream.findFirst().isPresent()) {
                throw new MountPointNotEmptyDirectoryException(toCheck, "Mountpoint directory is not empty: " + String.valueOf(toCheck));
            }
        }
    }

    @VisibleForTesting
    static Path getHideaway(Path mountPoint) {
        return mountPoint.resolveSibling(HIDEAWAY_PREFIX + mountPoint.getFileName().toString() + HIDEAWAY_SUFFIX);
    }

    @VisibleForTesting
    static enum MountPointState {
        NOT_EXISTING,
        EMPTY_DIR,
        BROKEN_JUNCTION;

    }
}

