/*
 * Decompiled with CFR 0.152.
 */
package org.cryptomator.windows.keychain;

import com.fasterxml.jackson.core.JacksonException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Reader;
import java.io.Writer;
import java.nio.ByteBuffer;
import java.nio.CharBuffer;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.cryptomator.integrations.common.OperatingSystem;
import org.cryptomator.integrations.common.Priority;
import org.cryptomator.integrations.keychain.KeychainAccessException;
import org.cryptomator.integrations.keychain.KeychainAccessProvider;
import org.cryptomator.windows.common.Localization;
import org.cryptomator.windows.keychain.KeychainEntry;
import org.cryptomator.windows.keychain.WinDataProtection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Priority(value=1000)
@OperatingSystem(value=OperatingSystem.Value.WINDOWS)
public class WindowsProtectedKeychainAccess
implements KeychainAccessProvider {
    private static final String KEYCHAIN_PATHS_PROPERTY = "cryptomator.integrationsWin.keychainPaths";
    private static final Logger LOG = LoggerFactory.getLogger(WindowsProtectedKeychainAccess.class);
    private static final Path USER_HOME_REL = Path.of("~", new String[0]);
    private static final Path USER_HOME = Path.of(System.getProperty("user.home"), new String[0]);
    private static final ObjectMapper JSON_MAPPER = new ObjectMapper();
    private final List<Path> keychainPaths;
    private final WinDataProtection dataProtection;
    private Map<String, KeychainEntry> keychainEntries;

    public WindowsProtectedKeychainAccess() {
        this(WindowsProtectedKeychainAccess.readKeychainPathsFromEnv(), new WinDataProtection());
    }

    WindowsProtectedKeychainAccess(List<Path> keychainPaths, WinDataProtection dataProtection) {
        this.keychainPaths = keychainPaths;
        this.dataProtection = dataProtection;
    }

    private static List<Path> readKeychainPathsFromEnv() {
        String keychainPaths = System.getProperty(KEYCHAIN_PATHS_PROPERTY, "");
        return WindowsProtectedKeychainAccess.parsePaths(keychainPaths, System.getProperty("path.separator"));
    }

    static List<Path> parsePaths(String listOfPaths, String pathSeparator) {
        return Arrays.stream(listOfPaths.split(pathSeparator)).filter(Predicate.not(String::isEmpty)).map(x$0 -> Path.of(x$0, new String[0])).map(WindowsProtectedKeychainAccess::resolveHomeDir).collect(Collectors.toList());
    }

    private static Path resolveHomeDir(Path path) {
        if (path.startsWith(USER_HOME_REL)) {
            return USER_HOME.resolve(USER_HOME_REL.relativize(path));
        }
        return path;
    }

    public String displayName() {
        return Localization.get().getString("org.cryptomator.windows.keychain.displayName");
    }

    public void storePassphrase(String key, String displayName, CharSequence passphrase, boolean ignored) throws KeychainAccessException {
        this.loadKeychainEntriesIfNeeded();
        ByteBuffer buf = StandardCharsets.UTF_8.encode(CharBuffer.wrap(passphrase));
        byte[] cleartext = new byte[buf.remaining()];
        buf.get(cleartext);
        byte[] salt = this.generateSalt();
        byte[] ciphertext = this.dataProtection.protect(cleartext, salt);
        Arrays.fill(buf.array(), (byte)0);
        Arrays.fill(cleartext, (byte)0);
        this.keychainEntries.put(key, new KeychainEntry(ciphertext, salt));
        this.saveKeychainEntries();
    }

    public char[] loadPassphrase(String key) throws KeychainAccessException {
        this.loadKeychainEntriesIfNeeded();
        KeychainEntry entry = this.keychainEntries.get(key);
        if (entry == null) {
            return null;
        }
        byte[] cleartext = this.dataProtection.unprotect(entry.ciphertext(), entry.salt());
        if (cleartext == null) {
            return null;
        }
        CharBuffer buf = StandardCharsets.UTF_8.decode(ByteBuffer.wrap(cleartext));
        char[] passphrase = new char[buf.remaining()];
        buf.get(passphrase);
        Arrays.fill(cleartext, (byte)0);
        Arrays.fill(buf.array(), '\u0000');
        return passphrase;
    }

    public void deletePassphrase(String key) throws KeychainAccessException {
        this.loadKeychainEntriesIfNeeded();
        this.keychainEntries.remove(key);
        this.saveKeychainEntries();
    }

    public void changePassphrase(String key, String displayName, CharSequence passphrase) throws KeychainAccessException {
        this.loadKeychainEntriesIfNeeded();
        if (this.keychainEntries.remove(key) != null) {
            this.storePassphrase(key, passphrase);
        }
    }

    public boolean isSupported() {
        return !this.keychainPaths.isEmpty();
    }

    public boolean isLocked() {
        return false;
    }

    private byte[] generateSalt() {
        byte[] result = new byte[16];
        UUID uuid = UUID.randomUUID();
        ByteBuffer buf = ByteBuffer.wrap(result);
        buf.putLong(uuid.getMostSignificantBits());
        buf.putLong(uuid.getLeastSignificantBits());
        return result;
    }

    private void loadKeychainEntriesIfNeeded() throws KeychainAccessException {
        if (this.keychainEntries == null) {
            for (Path keychainPath : this.keychainPaths) {
                Optional<Map<String, KeychainEntry>> keychain = this.loadKeychainEntries(keychainPath);
                if (!keychain.isPresent()) continue;
                this.keychainEntries = keychain.get();
                break;
            }
        }
        if (this.keychainEntries == null) {
            LOG.info("Unable to load existing keychain file, creating new keychain.");
            this.keychainEntries = new HashMap<String, KeychainEntry>();
        }
    }

    /*
     * Enabled aggressive exception aggregation
     */
    Optional<Map<String, KeychainEntry>> loadKeychainEntries(Path keychainPath) throws KeychainAccessException {
        LOG.debug("Attempting to load keychain from {}", (Object)keychainPath);
        TypeReference<Map<String, KeychainEntry>> type = new TypeReference<Map<String, KeychainEntry>>(this){};
        try (InputStream in = Files.newInputStream(keychainPath, StandardOpenOption.READ);){
            Optional<Map<String, KeychainEntry>> optional;
            try (InputStreamReader reader = new InputStreamReader(in, StandardCharsets.UTF_8);){
                optional = Optional.ofNullable((Map)JSON_MAPPER.readValue((Reader)reader, (TypeReference)type));
            }
            return optional;
        }
        catch (NoSuchFileException e) {
            return Optional.empty();
        }
        catch (JacksonException je) {
            LOG.warn("Unable to parse keychain file, overwriting existing one.");
            return Optional.empty();
        }
        catch (IOException e) {
            throw new KeychainAccessException("Could not read keychain from path " + String.valueOf(keychainPath), (Throwable)e);
        }
    }

    private void saveKeychainEntries() throws KeychainAccessException {
        if (this.keychainPaths.isEmpty()) {
            throw new IllegalStateException("Can't save keychain if no keychain path is specified.");
        }
        this.saveKeychainEntries(this.keychainPaths.get(0));
    }

    private void saveKeychainEntries(Path keychainPath) throws KeychainAccessException {
        try (OutputStream out = Files.newOutputStream(keychainPath, StandardOpenOption.WRITE, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
             OutputStreamWriter writer = new OutputStreamWriter(out, StandardCharsets.UTF_8);){
            JSON_MAPPER.writeValue((Writer)writer, this.keychainEntries);
        }
        catch (IOException e) {
            throw new KeychainAccessException("Could not read keychain from path " + String.valueOf(keychainPath), (Throwable)e);
        }
    }
}

