/*
 * Decompiled with CFR 0.152.
 */
package com.android.tools.idea.rendering.imagepool;

import com.android.tools.idea.rendering.imagepool.ImagePool;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.FinalizablePhantomReference;
import com.google.common.base.FinalizableReferenceQueue;
import com.google.common.collect.EvictingQueue;
import com.google.common.collect.ForwardingQueue;
import com.google.common.collect.Sets;
import com.intellij.openapi.diagnostic.Logger;
import java.awt.AlphaComposite;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GraphicsConfiguration;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferInt;
import java.awt.image.WritableRaster;
import java.lang.ref.Reference;
import java.lang.ref.SoftReference;
import java.util.Arrays;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.NoSuchElementException;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

class ImagePoolImpl
implements ImagePool {
    private static final Logger LOG = Logger.getInstance(ImagePoolImpl.class);
    private static final Bucket NULL_BUCKET = new Bucket(0, 0, 0);
    private final int[] myBucketSizes;
    private final HashMap<String, Bucket> myPool = new HashMap();
    private final IdentityHashMap<Bucket, BucketStatsImpl> myBucketStats = new IdentityHashMap();
    private final BiFunction<Integer, Integer, Function<Integer, Integer>> myBucketSizingPolicy;
    private final FinalizableReferenceQueue myFinalizableReferenceQueue = new FinalizableReferenceQueue();
    private final Set<Reference<?>> myReferences = Sets.newConcurrentHashSet();
    private final LongAdder myTotalAllocatedBytes = new LongAdder();
    private final LongAdder myTotalInUseBytes = new LongAdder();
    private final ImagePool.Stats myStats = new ImagePool.Stats(){

        @Override
        public long totalBytesAllocated() {
            return ImagePoolImpl.this.myTotalAllocatedBytes.sum();
        }

        @Override
        public long totalBytesInUse() {
            return ImagePoolImpl.this.myTotalInUseBytes.sum();
        }

        @Override
        public ImagePool.BucketStats[] getBucketStats() {
            return (ImagePool.BucketStats[])ImagePoolImpl.this.myBucketStats.values().stream().toArray(ImagePool.BucketStats[]::new);
        }
    };
    private boolean isDisposed = false;

    ImagePoolImpl(@NotNull int[] bucketSizes, @NotNull BiFunction<Integer, Integer, Function<Integer, Integer>> bucketSizingPolicy) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("New ImagePool " + Arrays.toString(bucketSizes));
        }
        this.myBucketSizes = bucketSizes;
        Arrays.sort(this.myBucketSizes);
        this.myBucketSizingPolicy = bucketSizingPolicy;
    }

    @NotNull
    private static String getPoolKey(int w, int h, int type) {
        return "" + w + 'x' + h + '-' + type;
    }

    @NotNull
    private Bucket getTypeBucket(int w, int h, int type) {
        if (this.myBucketSizingPolicy.apply(w, h).apply(type) == 0) {
            return NULL_BUCKET;
        }
        int widthBucket = -1;
        int heightBucket = -1;
        for (int bucketMinSize : this.myBucketSizes) {
            if (widthBucket == -1 && w < bucketMinSize) {
                widthBucket = bucketMinSize;
                if (heightBucket != -1) break;
            }
            if (heightBucket != -1 || h >= bucketMinSize) continue;
            heightBucket = bucketMinSize;
            if (widthBucket != -1) break;
        }
        if (widthBucket == -1 || heightBucket == -1) {
            return NULL_BUCKET;
        }
        String poolKey = ImagePoolImpl.getPoolKey(widthBucket, heightBucket, type);
        int finalWidthBucket = widthBucket;
        int finalHeightBucket = heightBucket;
        return this.myPool.computeIfAbsent(poolKey, k -> {
            int size = this.myBucketSizingPolicy.apply(finalWidthBucket, finalHeightBucket).apply(type);
            if (size == 0) {
                return NULL_BUCKET;
            }
            Bucket newBucket = new Bucket(finalWidthBucket, finalHeightBucket, size);
            this.myBucketStats.put(newBucket, new BucketStatsImpl(newBucket));
            return newBucket;
        });
    }

    @VisibleForTesting
    @NotNull
    ImageImpl create(final int w, final int h, final int type, final @Nullable Consumer<BufferedImage> freedCallback) {
        BufferedImage image;
        assert (!this.isDisposed) : "ImagePool already disposed";
        final Bucket bucket = this.getTypeBucket(w, h, type);
        final BucketStatsImpl bucketStats2 = this.myBucketStats.get((Object)bucket);
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("create(%dx%d-%d) in bucket (%dx%d) hasStats=%b\n", w, h, type, bucket.myMinWidth, bucket.myMinHeight, bucketStats2 != null));
        }
        try {
            SoftReference imageRef = (SoftReference)bucket.remove();
            while ((image = (BufferedImage)imageRef.get()) == null) {
                imageRef = (SoftReference)bucket.remove();
            }
            long totalSize = image.getWidth() * image.getHeight();
            if (bucketStats2 != null) {
                bucketStats2.bucketHit();
            }
            if (LOG.isDebugEnabled()) {
                double wasted = totalSize - (long)(w * h);
                LOG.debug(String.format("  Re-used image %dx%d - %d\n  pool buffer %dx%d\n  wasted %d%%\n", w, h, type, image.getWidth(), image.getHeight(), (int)(wasted / (double)totalSize * 100.0)));
            }
            this.myTotalInUseBytes.add(totalSize * 4L);
            if (image.getRaster().getDataBuffer().getDataType() == 3) {
                Arrays.fill(((DataBufferInt)image.getRaster().getDataBuffer()).getData(), 0);
            } else {
                Graphics2D g = image.createGraphics();
                g.setComposite(AlphaComposite.Clear);
                g.fillRect(0, 0, w, h);
                g.dispose();
            }
        }
        catch (NoSuchElementException e) {
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("  New image %dx%d - %d\n", w, h, type));
            }
            if (bucketStats2 != null) {
                bucketStats2.bucketMiss();
            }
            int newImageWidth = Math.max(bucket.myMinWidth, w);
            int newImageHeight = Math.max(bucket.myMinHeight, h);
            image = new BufferedImage(newImageWidth, newImageHeight, type);
            image.setAccelerationPriority(0.9f);
            long estimatedSize = newImageWidth * newImageHeight * 4;
            this.myTotalAllocatedBytes.add(estimatedSize);
            this.myTotalInUseBytes.add(estimatedSize);
        }
        ImageImpl pooledImage = new ImageImpl(w, h, image);
        final BufferedImage imagePointer = image;
        FinalizablePhantomReference<ImagePool.Image> reference2 = new FinalizablePhantomReference<ImagePool.Image>((ImagePool.Image)pooledImage, this.myFinalizableReferenceQueue){

            public void finalizeReferent() {
                if (ImagePoolImpl.this.myReferences.remove((Object)this)) {
                    boolean accepted = bucket.offer(new SoftReference<BufferedImage>(imagePointer));
                    if (bucketStats2 != null) {
                        if (accepted) {
                            bucketStats2.returnedImageAccepted();
                        } else {
                            bucketStats2.returnedImageRejected();
                        }
                    }
                    if (LOG.isDebugEnabled()) {
                        LOG.debug(String.format("%s image (%dx%d-%d) in bucket (%dx%d)\n", accepted ? "Released" : "Rejected", w, h, type, bucket.myMinWidth, bucket.myMinHeight));
                    }
                    long estimatedSize = imagePointer.getWidth() * imagePointer.getHeight() * 4;
                    if (!accepted) {
                        ImagePoolImpl.this.myTotalAllocatedBytes.add(-estimatedSize);
                    } else {
                        ImagePoolImpl.this.myTotalInUseBytes.add(-estimatedSize);
                    }
                    if (freedCallback != null) {
                        freedCallback.accept(imagePointer);
                    }
                }
            }
        };
        pooledImage.myOwnReference = (FinalizablePhantomReference)reference2;
        this.myReferences.add((Reference<?>)reference2);
        return pooledImage;
    }

    @Override
    @NotNull
    public ImagePool.Image create(int w, int h, int type) {
        return this.create(w, h, type, null);
    }

    @Override
    @NotNull
    public ImagePool.Image copyOf(@Nullable BufferedImage origin) {
        if (origin == null) {
            return ImagePool.NULL_POOLED_IMAGE;
        }
        int w = origin.getWidth();
        int h = origin.getHeight();
        int type = origin.getType();
        ImageImpl image = this.create(w, h, type, null);
        image.drawFrom(origin);
        return image;
    }

    @Override
    @Nullable
    public ImagePool.Stats getStats() {
        return this.myStats;
    }

    @Override
    public void dispose() {
        this.isDisposed = true;
        this.myFinalizableReferenceQueue.close();
        this.myReferences.clear();
        this.myPool.clear();
    }

    public static class ImageImpl
    implements ImagePool.Image {
        private static boolean ourTrackDisposeCall = ImageImpl.class.desiredAssertionStatus();
        private FinalizablePhantomReference<ImagePool.Image> myOwnReference = null;
        private ReadWriteLock myLock = new ReentrantReadWriteLock();
        private StackTraceElement[] myDisposeStackTrace = null;
        @VisibleForTesting
        @Nullable
        BufferedImage myBuffer;
        final int myWidth;
        final int myHeight;

        private ImageImpl(int w, int h, @NotNull BufferedImage image) {
            assert (w <= image.getWidth() && h <= image.getHeight());
            this.myWidth = w;
            this.myHeight = h;
            this.myBuffer = image;
        }

        @NotNull
        private static String stackTraceToAssertionString(@Nullable StackTraceElement[] trace) {
            if (trace == null) {
                return "Image was already disposed";
            }
            StringBuilder builder2 = new StringBuilder("Image was already disposed at: \n");
            int i2 = 0;
            for (StackTraceElement element2 : trace) {
                if (i2++ == 0) continue;
                builder2.append("\t\t").append(element2.toString()).append('\n');
            }
            return builder2.toString();
        }

        private void assertIfDisposed() {
            assert (this.myBuffer != null) : ImageImpl.stackTraceToAssertionString(this.myDisposeStackTrace);
        }

        @Override
        public int getWidth() {
            return this.myWidth;
        }

        @Override
        public int getHeight() {
            return this.myHeight;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void drawImageTo(@NotNull Graphics g, int dx1, int dy1, int dx2, int dy2, int sx1, int sy1, int sx2, int sy2) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                g.drawImage(this.myBuffer, dx1, dy1, dx2, dy2, sx1, sy1, sx2, sy2, null);
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void paint(@NotNull Consumer<Graphics2D> command) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                Graphics2D g = this.myBuffer.createGraphics();
                try {
                    command.accept(g);
                }
                finally {
                    g.dispose();
                }
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        @Nullable
        public BufferedImage getCopy(@Nullable GraphicsConfiguration gc, int x, int y, int w, int h) {
            this.myLock.readLock().lock();
            if (this.myBuffer == null) {
                LOG.debug("getCopy for already disposed image");
                return null;
            }
            try {
                if (x + w > this.myWidth) {
                    throw new IndexOutOfBoundsException(String.format("x (%d) + y (%d) is out bounds (image width is = %d)", x, y, this.myWidth));
                }
                if (y + h > this.myHeight) {
                    throw new IndexOutOfBoundsException(String.format("y (%d) + h (%d) is out bounds (image height is = %d)", y, h, this.myHeight));
                }
                BufferedImage newImage = gc != null ? gc.createCompatibleImage(w, h) : new BufferedImage(w, h, this.myBuffer.getType());
                Graphics2D g = newImage.createGraphics();
                try {
                    g.drawImage(this.myBuffer, 0, 0, w, h, x, y, x + w, y + h, null);
                }
                finally {
                    g.dispose();
                }
                BufferedImage bufferedImage = newImage;
                return bufferedImage;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        @Override
        @Nullable
        public BufferedImage getCopy() {
            this.myLock.readLock().lock();
            if (this.myBuffer == null) {
                LOG.debug("getCopy for already disposed image");
                return null;
            }
            try {
                WritableRaster raster = this.myBuffer.copyData(this.myBuffer.getRaster().createCompatibleWritableRaster(0, 0, this.myWidth, this.myHeight));
                BufferedImage bufferedImage = new BufferedImage(this.myBuffer.getColorModel(), raster, this.myBuffer.isAlphaPremultiplied(), null);
                return bufferedImage;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        @Override
        public void dispose() {
            this.assertIfDisposed();
            this.myLock.writeLock().lock();
            if (ourTrackDisposeCall) {
                this.myDisposeStackTrace = Thread.currentThread().getStackTrace();
            }
            try {
                this.myBuffer = null;
                if (this.myOwnReference != null) {
                    this.myOwnReference.finalizeReferent();
                }
            }
            finally {
                this.myLock.writeLock().unlock();
            }
        }

        @Override
        public boolean isValid() {
            this.myLock.readLock().lock();
            try {
                boolean bl = this.myBuffer != null;
                return bl;
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        void drawFrom(@NotNull BufferedImage origin) {
            this.assertIfDisposed();
            this.myLock.readLock().lock();
            try {
                Graphics g = this.myBuffer.getGraphics();
                try {
                    g.drawImage(origin, 0, 0, null);
                }
                finally {
                    g.dispose();
                }
            }
            finally {
                this.myLock.readLock().unlock();
            }
        }
    }

    private static class Bucket
    extends ForwardingQueue<SoftReference<BufferedImage>> {
        private final Queue<SoftReference<BufferedImage>> myDelegate;
        private final int myMinWidth;
        private final int myMinHeight;
        private final int myMaxSize;

        public Bucket(int minWidth, int minHeight, int maxSize) {
            if (maxSize == 0) {
                LOG.warn("0 maxSize for Bucket. This Bucket will not be used.");
            }
            this.myMinWidth = minWidth;
            this.myMinHeight = minHeight;
            this.myMaxSize = maxSize;
            this.myDelegate = maxSize == 0 ? EvictingQueue.create((int)0) : new ArrayBlockingQueue(maxSize);
        }

        protected Queue<SoftReference<BufferedImage>> delegate() {
            return this.myDelegate;
        }

        public int getMaxSize() {
            return this.myMaxSize;
        }
    }

    private static final class BucketStatsImpl
    implements ImagePool.BucketStats {
        private final Bucket myBucket;
        private final AtomicLong myLastAccessMs = new AtomicLong(System.currentTimeMillis());
        private final AtomicLong myBucketMiss = new AtomicLong(0L);
        private final AtomicLong myBucketHit = new AtomicLong(0L);
        private final AtomicLong myBucketFull = new AtomicLong(0L);
        private final AtomicLong myBucketHadSpace = new AtomicLong(0L);

        BucketStatsImpl(@NotNull Bucket bucket) {
            this.myBucket = bucket;
        }

        @Override
        public int getMinWidth() {
            return this.myBucket.myMinWidth;
        }

        @Override
        public int getMinHeight() {
            return this.myBucket.myMinHeight;
        }

        @Override
        public int maxSize() {
            return this.myBucket.getMaxSize();
        }

        @Override
        public long getLastAccessTimeMs() {
            return this.myLastAccessMs.get();
        }

        @Override
        public long bucketHits() {
            return this.myBucketHit.get();
        }

        @Override
        public long bucketMisses() {
            return this.myBucketMiss.get();
        }

        @Override
        public long bucketWasFull() {
            return this.myBucketFull.get();
        }

        @Override
        public long imageWasReturned() {
            return this.myBucketHadSpace.get();
        }

        public void bucketHit() {
            this.myLastAccessMs.set(System.currentTimeMillis());
            this.myBucketHit.incrementAndGet();
        }

        public void bucketMiss() {
            this.myLastAccessMs.set(System.currentTimeMillis());
            this.myBucketMiss.incrementAndGet();
        }

        public void returnedImageAccepted() {
            this.myBucketHadSpace.incrementAndGet();
        }

        public void returnedImageRejected() {
            this.myBucketFull.incrementAndGet();
        }
    }
}

