/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.util.io;

import com.intellij.UtilBundle;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.util.Condition;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileAttributes;
import com.intellij.openapi.util.io.FileSystemUtil;
import com.intellij.openapi.util.io.FileTooBigException;
import com.intellij.openapi.util.io.FileUtilRt;
import com.intellij.openapi.util.io.OSAgnosticPathUtil;
import com.intellij.openapi.util.text.StringUtil;
import com.intellij.openapi.util.text.Strings;
import com.intellij.util.ArrayUtil;
import com.intellij.util.ArrayUtilRt;
import com.intellij.util.ObjectUtils;
import com.intellij.util.Processor;
import com.intellij.util.SystemProperties;
import com.intellij.util.ThreeState;
import com.intellij.util.concurrency.AppExecutorUtil;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.containers.JBIterable;
import com.intellij.util.containers.JBTreeTraverser;
import com.intellij.util.io.URLUtil;
import com.intellij.util.text.FilePathHashingStrategy;
import com.intellij.util.text.StringFactory;
import gnu.trove.TObjectHashingStrategy;
import java.io.BufferedReader;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.io.Reader;
import java.net.MalformedURLException;
import java.net.URI;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.InvalidPathException;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.PosixFileAttributeView;
import java.nio.file.attribute.PosixFilePermission;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Future;
import java.util.regex.Pattern;
import org.intellij.lang.annotations.RegExp;
import org.jetbrains.annotations.ApiStatus;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@ApiStatus.NonExtendable
public class FileUtil
extends FileUtilRt {
    public static final String ASYNC_DELETE_EXTENSION = ".__del__";
    public static final int REGEX_PATTERN_FLAGS = SystemInfo.isFileSystemCaseSensitive ? 0 : 2;
    public static final TObjectHashingStrategy<String> PATH_HASHING_STRATEGY = FilePathHashingStrategy.create();
    public static final TObjectHashingStrategy<CharSequence> PATH_CHAR_SEQUENCE_HASHING_STRATEGY = FilePathHashingStrategy.createForCharSequence();
    public static final TObjectHashingStrategy<File> FILE_HASHING_STRATEGY = new TObjectHashingStrategy<File>(){

        public int computeHashCode(File object) {
            return FileUtil.fileHashCode(object);
        }

        public boolean equals(File o1, File o2) {
            return FileUtil.filesEqual(o1, o2);
        }
    };
    private static final Logger LOG = Logger.getInstance(FileUtil.class);
    private static final FileUtilRt.SymlinkResolver SYMLINK_RESOLVER = new FileUtilRt.SymlinkResolver(){

        @Override
        @NotNull
        public String resolveSymlinksAndCanonicalize(@NotNull String path, char separatorChar, boolean removeLastSlash) {
            try {
                return new File(path).getCanonicalPath().replace(separatorChar, '/');
            }
            catch (IOException ignore) {
                return FileUtil.toCanonicalPath(path, separatorChar, removeLastSlash, false);
            }
        }

        @Override
        public boolean isSymlink(@NotNull CharSequence path) {
            return FileSystemUtil.isSymLink(new File(path.toString()));
        }
    };

    @NotNull
    public static String join(String ... parts) {
        return String.join((CharSequence)File.separator, parts);
    }

    @Nullable
    public static String getRelativePath(File base, File file) {
        return FileUtilRt.getRelativePath(base, file);
    }

    @Nullable
    public static String getRelativePath(@NotNull String basePath, @NotNull String filePath, char separator) {
        return FileUtilRt.getRelativePath(basePath, filePath, separator);
    }

    @Nullable
    public static String getRelativePath(@NotNull String basePath, @NotNull String filePath, char separator, boolean caseSensitive) {
        return FileUtilRt.getRelativePath(basePath, filePath, separator, caseSensitive);
    }

    public static boolean isAbsolute(@NotNull String path) {
        return !path.isEmpty() && new File(path).isAbsolute();
    }

    public static boolean exists(@Nullable String path) {
        return path != null && new File(path).exists();
    }

    public static boolean isAncestor(@NotNull File ancestor, @NotNull File file, boolean strict) {
        return FileUtil.isAncestor(ancestor.getPath(), file.getPath(), strict);
    }

    public static boolean isAncestor(@NotNull String ancestor, @NotNull String file, boolean strict) {
        return !ThreeState.NO.equals((Object)FileUtil.isAncestorThreeState(ancestor, file, strict));
    }

    @NotNull
    public static ThreeState isAncestorThreeState(@NotNull String ancestor, @NotNull String file, boolean strict) {
        String ancestorPath = FileUtil.toCanonicalPath(ancestor);
        String filePath = FileUtil.toCanonicalPath(file);
        return FileUtil.startsWith(filePath, ancestorPath, strict, SystemInfo.isFileSystemCaseSensitive, true);
    }

    public static boolean startsWith(@NotNull String path, @NotNull String start) {
        return FileUtil.startsWith(path, start, SystemInfo.isFileSystemCaseSensitive);
    }

    public static boolean startsWith(@NotNull String path, @NotNull String start, boolean caseSensitive) {
        return FileUtil.startsWith(path, start, caseSensitive, false);
    }

    public static boolean startsWith(@NotNull String path, @NotNull String start, boolean caseSensitive, boolean strict) {
        return !ThreeState.NO.equals((Object)FileUtil.startsWith(path, start, strict, caseSensitive, false));
    }

    private static ThreeState startsWith(String path, String prefix, boolean strict, boolean caseSensitive, boolean checkImmediateParent) {
        char next;
        int pathLength = path.length();
        int prefixLength = prefix.length();
        if (prefixLength == 0) {
            return pathLength == 0 ? ThreeState.YES : ThreeState.UNSURE;
        }
        if (prefixLength > pathLength) {
            return ThreeState.NO;
        }
        if (!path.regionMatches(!caseSensitive, 0, prefix, 0, prefixLength)) {
            return ThreeState.NO;
        }
        if (pathLength == prefixLength) {
            return strict ? ThreeState.NO : ThreeState.YES;
        }
        char lastPrefixChar = prefix.charAt(prefixLength - 1);
        int slashOrSeparatorIdx = prefixLength;
        if (lastPrefixChar == '/' || lastPrefixChar == File.separatorChar) {
            slashOrSeparatorIdx = prefixLength - 1;
        }
        if ((next = path.charAt(slashOrSeparatorIdx)) == '/' || next == File.separatorChar) {
            if (!checkImmediateParent) {
                return ThreeState.YES;
            }
            if (slashOrSeparatorIdx == pathLength - 1) {
                return ThreeState.YES;
            }
            int idxNext = path.indexOf(next, slashOrSeparatorIdx + 1);
            idxNext = idxNext == -1 ? path.indexOf(next == '/' ? 92 : 47, slashOrSeparatorIdx + 1) : idxNext;
            return idxNext == -1 ? ThreeState.YES : ThreeState.UNSURE;
        }
        return ThreeState.NO;
    }

    @Nullable
    public static File findAncestor(@NotNull File f1, @NotNull File f2) {
        File ancestor;
        for (ancestor = f1; ancestor != null && !FileUtil.isAncestor(ancestor, f2, false); ancestor = ancestor.getParentFile()) {
        }
        return ancestor;
    }

    @Nullable
    public static File getParentFile(@NotNull File file) {
        return FileUtilRt.getParentFile(file);
    }

    public static byte @NotNull [] loadFileBytes(@NotNull File file) throws IOException {
        byte[] bytes;
        try (FileInputStream stream = new FileInputStream(file);){
            long len = file.length();
            if (len < 0L) {
                throw new IOException("File length reported negative, probably doesn't exist");
            }
            if (FileUtil.isTooLarge(len)) {
                throw new FileTooBigException("Attempt to load '" + file + "' in memory buffer, file length is " + len + " bytes.");
            }
            bytes = FileUtil.loadBytes(stream, (int)len);
        }
        return bytes;
    }

    public static byte @NotNull [] loadFirstAndClose(@NotNull InputStream stream, int maxLength) throws IOException {
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        try {
            FileUtil.copy(stream, maxLength, (OutputStream)buffer);
        }
        finally {
            stream.close();
        }
        return buffer.toByteArray();
    }

    @NotNull
    public static String loadTextAndClose(@NotNull InputStream stream) throws IOException {
        return FileUtil.loadTextAndClose(new InputStreamReader(stream, StandardCharsets.UTF_8));
    }

    @NotNull
    public static String loadTextAndClose(@NotNull Reader reader) throws IOException {
        try {
            String string = StringFactory.createShared(FileUtil.adaptiveLoadText(reader));
            return string;
        }
        finally {
            reader.close();
        }
    }

    public static char @NotNull [] adaptiveLoadText(@NotNull Reader reader) throws IOException {
        int n;
        char[] chars = new char[4096];
        ArrayList<char[]> buffers = null;
        int count = 0;
        int total = 0;
        while ((n = reader.read(chars, count, chars.length - count)) > 0) {
            count += n;
            if (total > 0xA00000) {
                throw new FileTooBigException("File too big " + reader);
            }
            total += n;
            if (count != chars.length) continue;
            if (buffers == null) {
                buffers = new ArrayList<char[]>();
            }
            buffers.add(chars);
            int newLength = Math.min(0x100000, chars.length * 2);
            chars = new char[newLength];
            count = 0;
        }
        char[] result = new char[total];
        if (buffers != null) {
            for (char[] buffer : buffers) {
                System.arraycopy(buffer, 0, result, result.length - total, buffer.length);
                total -= buffer.length;
            }
        }
        System.arraycopy(chars, 0, result, result.length - total, total);
        return result;
    }

    public static byte @NotNull [] adaptiveLoadBytes(@NotNull InputStream stream) throws IOException {
        int n;
        byte[] bytes = FileUtil.getThreadLocalBuffer();
        ArrayList<byte[]> buffers = null;
        int count = 0;
        int total = 0;
        while ((n = stream.read(bytes, count, bytes.length - count)) > 0) {
            count += n;
            if (total > 0xA00000) {
                throw new FileTooBigException("File too big " + stream);
            }
            total += n;
            if (count != bytes.length) continue;
            if (buffers == null) {
                buffers = new ArrayList<byte[]>();
            }
            buffers.add(bytes);
            int newLength = Math.min(0x100000, bytes.length * 2);
            bytes = new byte[newLength];
            count = 0;
        }
        byte[] result = new byte[total];
        if (buffers != null) {
            for (byte[] buffer : buffers) {
                System.arraycopy(buffer, 0, result, result.length - total, buffer.length);
                total -= buffer.length;
            }
        }
        System.arraycopy(bytes, 0, result, result.length - total, total);
        return result;
    }

    @NotNull
    public static Future<Void> asyncDelete(@NotNull File file) {
        return FileUtil.asyncDelete(Collections.singleton(file));
    }

    @NotNull
    public static Future<Void> asyncDelete(@NotNull Collection<? extends File> files) {
        ArrayList<File> tempFiles = new ArrayList<File>();
        for (File file : files) {
            File tempFile = FileUtil.renameToTempFileOrDelete(file);
            if (tempFile == null) continue;
            tempFiles.add(tempFile);
        }
        return tempFiles.isEmpty() ? CompletableFuture.completedFuture(null) : AppExecutorUtil.getAppExecutorService().submit(() -> {
            Thread currentThread = Thread.currentThread();
            int priority = currentThread.getPriority();
            currentThread.setPriority(1);
            try {
                for (File tempFile : tempFiles) {
                    FileUtil.delete(tempFile);
                }
            }
            finally {
                currentThread.setPriority(priority);
            }
            return null;
        });
    }

    @Nullable
    private static File renameToTempFileOrDelete(@NotNull File file) {
        String originalFileName;
        File tempFile;
        String tempDir = FileUtil.getTempDirectory();
        boolean isSameDrive = true;
        if (SystemInfo.isWindows) {
            String tempDirDrive = tempDir.substring(0, 2);
            String fileDrive = file.getAbsolutePath().substring(0, 2);
            isSameDrive = tempDirDrive.equalsIgnoreCase(fileDrive);
        }
        if (isSameDrive && file.renameTo(tempFile = FileUtil.getTempFile(originalFileName = file.getName(), tempDir))) {
            return tempFile;
        }
        FileUtil.delete(file);
        return null;
    }

    @NotNull
    private static File getTempFile(@NotNull String originalFileName, @NotNull String parent) {
        int randomSuffix;
        int i = randomSuffix = (int)(System.currentTimeMillis() % 1000L);
        String name;
        File tempFile;
        while ((tempFile = new File(parent, name = "___" + originalFileName + i + ASYNC_DELETE_EXTENSION)).exists()) {
            ++i;
        }
        return tempFile;
    }

    public static boolean delete(@NotNull File file) {
        return FileUtilRt.delete(file);
    }

    public static void delete(@NotNull Path path) throws IOException {
        FileUtilRt.deleteRecursivelyNIO(path);
    }

    public static boolean createParentDirs(@NotNull File file) {
        return FileUtilRt.createParentDirs(file);
    }

    public static boolean createDirectory(@NotNull File path) {
        return FileUtilRt.createDirectory(path);
    }

    public static boolean createIfDoesntExist(@NotNull File file) {
        return FileUtilRt.createIfNotExists(file);
    }

    public static boolean ensureCanCreateFile(@NotNull File file) {
        return FileUtilRt.ensureCanCreateFile(file);
    }

    public static void copy(@NotNull File fromFile, @NotNull File toFile) throws IOException {
        FileUtil.performCopy(fromFile, toFile, true);
    }

    public static void copyContent(@NotNull File fromFile, @NotNull File toFile) throws IOException {
        FileUtil.performCopy(fromFile, toFile, false);
    }

    private static void performCopy(@NotNull File fromFile, @NotNull File toFile, boolean syncTimestamp) throws IOException {
        if (FileUtil.filesEqual(fromFile, toFile)) {
            return;
        }
        try (FileOutputStream fos = FileUtil.openOutputStream(toFile);
             FileInputStream fis = new FileInputStream(fromFile);){
            FileUtil.copy(fis, fos);
        }
        if (syncTimestamp) {
            long timeStamp = fromFile.lastModified();
            if (timeStamp < 0L) {
                LOG.warn("Invalid timestamp " + timeStamp + " of '" + fromFile + "'");
            } else if (!toFile.setLastModified(timeStamp)) {
                LOG.warn("Unable to set timestamp " + timeStamp + " to '" + toFile + "'");
            }
        }
        if (SystemInfo.isUnix && fromFile.canExecute()) {
            FileSystemUtil.clonePermissionsToExecute(fromFile.getPath(), toFile.getPath());
        }
    }

    @NotNull
    private static FileOutputStream openOutputStream(@NotNull File file) throws IOException {
        try {
            return new FileOutputStream(file);
        }
        catch (FileNotFoundException e) {
            File parentFile = file.getParentFile();
            if (parentFile == null) {
                throw new IOException("Parent file is null for " + file.getPath(), e);
            }
            FileUtil.createParentDirs(file);
            return new FileOutputStream(file);
        }
    }

    public static void copy(@NotNull InputStream inputStream, @NotNull OutputStream outputStream) throws IOException {
        FileUtilRt.copy(inputStream, outputStream);
    }

    public static void copy(@NotNull InputStream inputStream, int maxSize, @NotNull OutputStream outputStream) throws IOException {
        FileUtil.copy(inputStream, (long)maxSize, outputStream);
    }

    public static void copy(@NotNull InputStream inputStream, long maxSize, @NotNull OutputStream outputStream) throws IOException {
        int read;
        byte[] buffer = FileUtil.getThreadLocalBuffer();
        for (long toRead = maxSize; toRead > 0L && (read = inputStream.read(buffer, 0, (int)Math.min((long)buffer.length, toRead))) >= 0; toRead -= (long)read) {
            outputStream.write(buffer, 0, read);
        }
    }

    public static void copyFileOrDir(@NotNull File from, @NotNull File to) throws IOException {
        FileUtil.copyFileOrDir(from, to, from.isDirectory());
    }

    public static void copyFileOrDir(@NotNull File from, @NotNull File to, boolean isDir) throws IOException {
        if (isDir) {
            FileUtil.copyDir(from, to);
        } else {
            FileUtil.copy(from, to);
        }
    }

    public static void copyDir(@NotNull File fromDir, @NotNull File toDir) throws IOException {
        FileUtil.copyDir(fromDir, toDir, true);
    }

    public static void copyDirContent(@NotNull File fromDir, @NotNull File toDir) throws IOException {
        File[] children;
        for (File child : children = ObjectUtils.notNull(fromDir.listFiles(), ArrayUtilRt.EMPTY_FILE_ARRAY)) {
            FileUtil.copyFileOrDir(child, new File(toDir, child.getName()));
        }
    }

    public static void copyDir(@NotNull File fromDir, @NotNull File toDir, boolean copySystemFiles) throws IOException {
        FileUtil.copyDir(fromDir, toDir, copySystemFiles ? null : file -> !StringUtil.startsWithChar(file.getName(), '.'));
    }

    public static void copyDir(@NotNull File fromDir, @NotNull File toDir, @Nullable FileFilter filter) throws IOException {
        FileUtil.ensureExists(toDir);
        if (FileUtil.isAncestor(fromDir, toDir, true)) {
            LOG.error(fromDir.getAbsolutePath() + " is ancestor of " + toDir + ". Can't copy to itself.");
            return;
        }
        File[] files = fromDir.listFiles();
        if (files == null) {
            throw new IOException(UtilBundle.message("exception.directory.is.invalid", fromDir.getPath()));
        }
        if (!fromDir.canRead()) {
            throw new IOException(UtilBundle.message("exception.directory.is.not.readable", fromDir.getPath()));
        }
        for (File file : files) {
            if (filter != null && !filter.accept(file)) continue;
            if (file.isDirectory()) {
                FileUtil.copyDir(file, new File(toDir, file.getName()), filter);
                continue;
            }
            FileUtil.copy(file, new File(toDir, file.getName()));
        }
    }

    public static void ensureExists(@NotNull File dir) throws IOException {
        if (!dir.exists() && !dir.mkdirs()) {
            throw new IOException(UtilBundle.message("exception.directory.can.not.create", dir.getPath()));
        }
    }

    @NotNull
    public static String getNameWithoutExtension(@NotNull File file) {
        return FileUtilRt.getNameWithoutExtension(file.getName());
    }

    @NotNull
    public static String getNameWithoutExtension(@NotNull String name) {
        return FileUtilRt.getNameWithoutExtension(name);
    }

    @NotNull
    public static String createSequentFileName(@NotNull File aParentFolder, @NotNull String aFilePrefix, @NotNull String aExtension) {
        return FileUtil.findSequentNonexistentFile(aParentFolder, aFilePrefix, aExtension).getName();
    }

    @NotNull
    public static String createSequentFileName(@NotNull File aParentFolder, @NotNull String aFilePrefix, @NotNull String aExtension, @NotNull Condition<? super File> condition) {
        return FileUtil.findSequentFile(aParentFolder, aFilePrefix, aExtension, condition).getName();
    }

    @NotNull
    public static File findSequentNonexistentFile(@NotNull File parentFolder, @NotNull String filePrefix, @NotNull String extension) {
        return FileUtil.findSequentFile(parentFolder, filePrefix, extension, file -> !file.exists());
    }

    @NotNull
    public static File findSequentFile(@NotNull File parentFolder, @NotNull String filePrefix, @NotNull String extension, @NotNull Condition<? super File> condition) {
        int postfix = 0;
        String ext = extension.isEmpty() ? "" : '.' + extension;
        File candidate = new File(parentFolder, filePrefix + ext);
        while (!condition.value(candidate)) {
            candidate = new File(parentFolder, filePrefix + ++postfix + ext);
        }
        return candidate;
    }

    @NotNull
    public static String toSystemDependentName(@NotNull String aFileName) {
        return FileUtilRt.toSystemDependentName(aFileName);
    }

    @NotNull
    public static String toSystemIndependentName(@NotNull String aFileName) {
        return FileUtilRt.toSystemIndependentName(aFileName);
    }

    @Contract(value="null -> null")
    public static String toCanonicalPath(@Nullable String path) {
        return FileUtil.toCanonicalPath(path, File.separatorChar, true);
    }

    @Contract(value="null, _ -> null; !null,_->!null")
    public static String toCanonicalPath(@Nullable String path, boolean resolveSymlinksIfNecessary) {
        return FileUtil.toCanonicalPath(path, File.separatorChar, true, resolveSymlinksIfNecessary);
    }

    @Contract(value="null, _ -> null; !null,_->!null")
    public static String toCanonicalPath(@Nullable String path, char separatorChar) {
        return FileUtil.toCanonicalPath(path, separatorChar, true);
    }

    @Contract(value="null -> null; !null->!null")
    public static String toCanonicalUriPath(@Nullable String path) {
        return FileUtil.toCanonicalPath(path, '/', false);
    }

    @Contract(value="null, _, _, _ -> null; !null,_,_,_->!null")
    private static String toCanonicalPath(@Nullable String path, char separatorChar, boolean removeLastSlash, boolean resolveSymlinks) {
        FileUtilRt.SymlinkResolver symlinkResolver = resolveSymlinks ? SYMLINK_RESOLVER : null;
        return FileUtil.toCanonicalPath(path, separatorChar, removeLastSlash, symlinkResolver);
    }

    @NotNull
    public static String normalize(@NotNull String path) {
        int start = 0;
        boolean separator = false;
        if (SystemInfo.isWindows) {
            if (path.startsWith("//")) {
                start = 2;
                separator = true;
            } else if (path.startsWith("\\\\")) {
                return FileUtil.normalizeTail(0, path, false);
            }
        }
        for (int i = start; i < path.length(); ++i) {
            char c = path.charAt(i);
            if (c == '/') {
                if (separator) {
                    return FileUtil.normalizeTail(i, path, true);
                }
                separator = true;
                continue;
            }
            if (c == '\\') {
                return FileUtil.normalizeTail(i, path, separator);
            }
            separator = false;
        }
        return path;
    }

    @NotNull
    private static String normalizeTail(int prefixEnd, @NotNull String path, boolean separator) {
        StringBuilder result = new StringBuilder(path.length());
        result.append(path, 0, prefixEnd);
        int start = prefixEnd;
        if (start == 0 && SystemInfo.isWindows && (path.startsWith("//") || path.startsWith("\\\\"))) {
            start = 2;
            result.append("//");
            separator = true;
        }
        for (int i = start; i < path.length(); ++i) {
            char c = path.charAt(i);
            if (c == '/' || c == '\\') {
                if (!separator) {
                    result.append('/');
                }
                separator = true;
                continue;
            }
            result.append(c);
            separator = false;
        }
        return result.toString();
    }

    @NotNull
    public static String unquote(@NotNull String urlString) {
        urlString = urlString.replace('/', File.separatorChar);
        return URLUtil.unescapePercentSequences(urlString);
    }

    public static boolean isFilePathAcceptable(@NotNull File root, @Nullable FileFilter fileFilter) {
        if (fileFilter == null) {
            return true;
        }
        File file = root;
        do {
            if (fileFilter.accept(file)) continue;
            return false;
        } while ((file = file.getParentFile()) != null);
        return true;
    }

    public static boolean rename(@NotNull File source, @NotNull String newName) throws IOException {
        File target = new File(source.getParent(), newName);
        if (!SystemInfo.isFileSystemCaseSensitive && newName.equalsIgnoreCase(source.getName())) {
            File intermediate = FileUtil.createTempFile(source.getParentFile(), source.getName(), ".tmp", false, false);
            return source.renameTo(intermediate) && intermediate.renameTo(target);
        }
        return source.renameTo(target);
    }

    public static void rename(@NotNull File source, @NotNull File target) throws IOException {
        if (source.renameTo(target)) {
            return;
        }
        if (!source.exists()) {
            return;
        }
        FileUtil.copy(source, target);
        FileUtil.delete(source);
    }

    public static boolean filesEqual(@Nullable File file1, @Nullable File file2) {
        return FileUtil.pathsEqual(file1 == null ? null : file1.getPath(), file2 == null ? null : file2.getPath());
    }

    public static boolean pathsEqual(@Nullable String path1, @Nullable String path2) {
        if (path1 == path2) {
            return true;
        }
        if (path1 == null || path2 == null) {
            return false;
        }
        path1 = FileUtil.toCanonicalPath(path1);
        path2 = FileUtil.toCanonicalPath(path2);
        return PATH_HASHING_STRATEGY.equals((Object)path1, (Object)path2);
    }

    public static boolean namesEqual(@Nullable String name1, @Nullable String name2) {
        if (name1 == name2) {
            return true;
        }
        if (name1 == null || name2 == null) {
            return false;
        }
        return PATH_HASHING_STRATEGY.equals((Object)name1, (Object)name2);
    }

    public static int compareFiles(@Nullable File file1, @Nullable File file2) {
        return FileUtil.comparePaths(file1 == null ? null : file1.getPath(), file2 == null ? null : file2.getPath());
    }

    public static int comparePaths(@Nullable String path1, @Nullable String path2) {
        return OSAgnosticPathUtil.COMPARATOR.compare(path1, path2);
    }

    public static int fileHashCode(@Nullable File file) {
        return FileUtil.pathHashCode(file == null ? null : file.getPath());
    }

    public static int pathHashCode(@Nullable String path) {
        return Strings.isEmpty(path) ? 0 : PATH_HASHING_STRATEGY.computeHashCode((Object)FileUtil.toCanonicalPath(path));
    }

    @Deprecated
    @NotNull
    public static String getExtension(@NotNull String fileName) {
        return Strings.toLowerCase(FileUtilRt.getExtension(fileName));
    }

    @NotNull
    public static String resolveShortWindowsName(@NotNull String path) throws IOException {
        try {
            return SystemInfo.isWindows && FileUtil.containsWindowsShortName(path) ? Paths.get(path, new String[0]).toRealPath(LinkOption.NOFOLLOW_LINKS).toString() : path;
        }
        catch (InvalidPathException e) {
            throw new IOException(e);
        }
    }

    public static boolean containsWindowsShortName(@NotNull String path) {
        if (path.indexOf(126) < 0) {
            return false;
        }
        path = FileUtil.toSystemIndependentName(path);
        int start = 0;
        while (start < path.length()) {
            int dot;
            int end = path.indexOf(47, start);
            if (end < 0) {
                end = path.length();
            }
            if ((dot = path.lastIndexOf(46, end)) < start) {
                dot = end;
            }
            if (dot - start > 2 && dot - start <= 8 && end - dot - 1 <= 3 && path.charAt(dot - 2) == '~' && Character.isDigit(path.charAt(dot - 1))) {
                return true;
            }
            start = end + 1;
        }
        return false;
    }

    @Nullable
    public static String extractRootPath(@NotNull String normalizedPath) {
        if (SystemInfo.isWindows) {
            int p1;
            if (normalizedPath.length() >= 2 && normalizedPath.charAt(1) == ':') {
                return StringUtil.toUpperCase(normalizedPath.substring(0, 2));
            }
            if (normalizedPath.startsWith("//") && (p1 = normalizedPath.indexOf(47, 2)) > 2) {
                int p2 = normalizedPath.indexOf(47, p1 + 1);
                if (p2 > p1 + 1) {
                    return normalizedPath.substring(0, p2);
                }
                if (p2 < 0) {
                    return normalizedPath;
                }
            }
        } else if (StringUtil.startsWithChar(normalizedPath, '/')) {
            return "/";
        }
        return null;
    }

    public static void collectMatchedFiles(@NotNull File root, @NotNull Pattern pattern, @NotNull List<? super File> outFiles) {
        FileUtil.collectMatchedFiles(root, root, pattern, outFiles);
    }

    private static void collectMatchedFiles(@NotNull File absoluteRoot, @NotNull File root, @NotNull Pattern pattern, @NotNull List<? super File> files) {
        File[] dirs = root.listFiles();
        if (dirs == null) {
            return;
        }
        for (File dir : dirs) {
            if (dir.isFile()) {
                String path;
                String relativePath = FileUtil.getRelativePath(absoluteRoot, dir);
                if (relativePath == null || !pattern.matcher(path = FileUtil.toSystemIndependentName(relativePath)).matches()) continue;
                files.add(dir);
                continue;
            }
            FileUtil.collectMatchedFiles(absoluteRoot, dir, pattern, files);
        }
    }

    @RegExp
    @NotNull
    public static String convertAntToRegexp(@NotNull String antPattern) {
        return FileUtil.convertAntToRegexp(antPattern, true);
    }

    @RegExp
    @NotNull
    public static String convertAntToRegexp(@NotNull String antPattern, boolean ignoreStartingSlash) {
        boolean isTrailingSlash;
        int start;
        StringBuilder builder = new StringBuilder();
        int asteriskCount = 0;
        boolean recursive = true;
        for (int idx = start = ignoreStartingSlash && (StringUtil.startsWithChar(antPattern, '/') || StringUtil.startsWithChar(antPattern, '\\')) ? 1 : 0; idx < antPattern.length(); ++idx) {
            char ch = antPattern.charAt(idx);
            if (ch == '*') {
                ++asteriskCount;
                continue;
            }
            boolean foundRecursivePattern = recursive && asteriskCount == 2 && (ch == '/' || ch == '\\');
            boolean asterisksFound = asteriskCount > 0;
            asteriskCount = 0;
            boolean bl = recursive = ch == '/' || ch == '\\';
            if (foundRecursivePattern) {
                builder.append("(?:[^/]+/)*?");
                continue;
            }
            if (asterisksFound) {
                builder.append("[^/]*?");
            }
            if (ch == '(' || ch == ')' || ch == '[' || ch == ']' || ch == '^' || ch == '$' || ch == '.' || ch == '{' || ch == '}' || ch == '+' || ch == '|') {
                builder.append('\\').append(ch);
                continue;
            }
            if (ch == '?') {
                builder.append("[^/]{1}");
                continue;
            }
            if (ch == '\\') {
                builder.append('/');
                continue;
            }
            builder.append(ch);
        }
        boolean bl = isTrailingSlash = builder.length() > 0 && builder.charAt(builder.length() - 1) == '/';
        if (asteriskCount == 0 && isTrailingSlash || recursive && asteriskCount == 2) {
            if (isTrailingSlash) {
                builder.setLength(builder.length() - 1);
            }
            if (builder.length() == 0) {
                builder.append(".*");
            } else {
                builder.append("(?:$|/.+)");
            }
        } else if (asteriskCount > 0) {
            builder.append("[^/]*?");
        }
        return builder.toString();
    }

    public static boolean moveDirWithContent(@NotNull File fromDir, @NotNull File toDir) {
        if (!toDir.exists()) {
            return fromDir.renameTo(toDir);
        }
        File[] files = fromDir.listFiles();
        if (files == null) {
            return false;
        }
        boolean success = true;
        for (File fromFile : files) {
            File toFile = new File(toDir, fromFile.getName());
            success = success && fromFile.renameTo(toFile);
        }
        fromDir.delete();
        return success;
    }

    @NotNull
    public static String sanitizeFileName(@NotNull String name) {
        return FileUtil.sanitizeFileName(name, true);
    }

    @NotNull
    public static String sanitizeFileName(@NotNull String name, boolean strict) {
        return FileUtil.sanitizeFileName(name, strict, "_");
    }

    @NotNull
    public static String sanitizeFileName(@NotNull String name, boolean strict, @NotNull String replacement) {
        StringBuilder result = null;
        int last = 0;
        int length = name.length();
        for (int i = 0; i < length; ++i) {
            char c = name.charAt(i);
            boolean appendReplacement = true;
            if (c > '\u0000' && c < '\u00ff') {
                if (strict ? Character.isLetterOrDigit(c) || c == '_' : Character.isJavaIdentifierPart(c) || c == ' ' || c == '@' || c == '-') {
                    continue;
                }
            } else {
                appendReplacement = false;
            }
            if (result == null) {
                result = new StringBuilder();
            }
            if (last < i) {
                result.append(name, last, i);
            }
            if (appendReplacement) {
                result.append(replacement);
            }
            last = i + 1;
        }
        if (result == null) {
            return name;
        }
        if (last < length) {
            result.append(name, last, length);
        }
        return result.toString();
    }

    public static boolean canExecute(@NotNull File file) {
        return file.canExecute();
    }

    public static boolean canWrite(@NotNull String path) {
        FileAttributes attributes = FileSystemUtil.getAttributes(path);
        return attributes != null && attributes.isWritable();
    }

    public static void setReadOnlyAttribute(@NotNull String path, boolean readOnlyFlag) {
        boolean writableFlag;
        boolean bl = writableFlag = !readOnlyFlag;
        if (!new File(path).setWritable(writableFlag, false) && FileUtil.canWrite(path) != writableFlag) {
            LOG.warn("Can't set writable attribute of '" + path + "' to '" + readOnlyFlag + "'");
        }
    }

    public static void appendToFile(@NotNull File file, @NotNull String text) throws IOException {
        FileUtil.writeToFile(file, text.getBytes(StandardCharsets.UTF_8), true);
    }

    public static void writeToFile(@NotNull File file, byte @NotNull [] text) throws IOException {
        FileUtil.writeToFile(file, text, false);
    }

    public static void writeToFile(@NotNull File file, @NotNull String text) throws IOException {
        FileUtil.writeToFile(file, text, false);
    }

    public static void writeToFile(@NotNull File file, @NotNull String text, boolean append) throws IOException {
        FileUtil.writeToFile(file, text.getBytes(StandardCharsets.UTF_8), append);
    }

    public static void writeToFile(@NotNull File file, byte @NotNull [] text, int off, int len) throws IOException {
        FileUtil.writeToFile(file, text, off, len, false);
    }

    public static void writeToFile(@NotNull File file, byte @NotNull [] text, boolean append) throws IOException {
        FileUtil.writeToFile(file, text, 0, text.length, append);
    }

    private static void writeToFile(@NotNull File file, byte @NotNull [] text, int off, int len, boolean append) throws IOException {
        FileUtil.createParentDirs(file);
        try (FileOutputStream stream = new FileOutputStream(file, append);){
            ((OutputStream)stream).write(text, off, len);
        }
    }

    @NotNull
    public static JBTreeTraverser<File> fileTraverser(@Nullable File root) {
        return (JBTreeTraverser)Lazy.FILE_TRAVERSER.withRoot(root);
    }

    public static boolean processFilesRecursively(@NotNull File root, @NotNull Processor<? super File> processor) {
        return FileUtil.fileTraverser(root).bfsTraversal().processEach(processor);
    }

    @Deprecated
    public static boolean processFilesRecursively(@NotNull File root, @NotNull Processor<? super File> processor, @Nullable Processor<? super File> directoryFilter) {
        LinkedList<File> queue = new LinkedList<File>();
        queue.add(root);
        while (!queue.isEmpty()) {
            File[] children;
            File file = (File)queue.removeFirst();
            if (!processor.process(file)) {
                return false;
            }
            if (directoryFilter != null && (!file.isDirectory() || !directoryFilter.process(file)) || (children = file.listFiles()) == null) continue;
            ContainerUtil.addAll(queue, children);
        }
        return true;
    }

    @Nullable
    public static File findFirstThatExist(String ... paths) {
        for (String path : paths) {
            File file;
            if (Strings.isEmptyOrSpaces(path) || !(file = new File(FileUtil.toSystemDependentName(path))).exists()) continue;
            return file;
        }
        return null;
    }

    @NotNull
    public static List<File> findFilesByMask(@NotNull Pattern pattern, @NotNull File dir) {
        ArrayList<File> found = new ArrayList<File>();
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (file.isDirectory()) {
                    found.addAll(FileUtil.findFilesByMask(pattern, file));
                    continue;
                }
                if (!pattern.matcher(file.getName()).matches()) continue;
                found.add(file);
            }
        }
        return found;
    }

    @NotNull
    public static List<File> findFilesOrDirsByMask(@NotNull Pattern pattern, @NotNull File dir) {
        ArrayList<File> found = new ArrayList<File>();
        File[] files = dir.listFiles();
        if (files != null) {
            for (File file : files) {
                if (pattern.matcher(file.getName()).matches()) {
                    found.add(file);
                }
                if (!file.isDirectory()) continue;
                found.addAll(FileUtil.findFilesOrDirsByMask(pattern, file));
            }
        }
        return found;
    }

    @Nullable
    public static String findFileInProvidedPath(String providedPath, String ... fileNames) {
        File file;
        if (Strings.isEmpty(providedPath)) {
            return "";
        }
        File providedFile = new File(providedPath);
        if (providedFile.exists() && ArrayUtil.indexOf(fileNames, providedFile.getName()) >= 0) {
            return FileUtil.toSystemDependentName(providedFile.getPath());
        }
        if (providedFile.isDirectory()) {
            for (String fileName : fileNames) {
                file = new File(providedFile, fileName);
                if (!fileName.equals(file.getName()) || !file.exists()) continue;
                return FileUtil.toSystemDependentName(file.getPath());
            }
        }
        if ((providedFile = providedFile.getParentFile()) != null && providedFile.exists()) {
            for (String fileName : fileNames) {
                file = new File(providedFile, fileName);
                if (!fileName.equals(file.getName()) || !file.exists()) continue;
                return FileUtil.toSystemDependentName(file.getPath());
            }
        }
        return null;
    }

    @Deprecated
    public static boolean isAbsolutePlatformIndependent(@NotNull String path) {
        return FileUtil.isUnixAbsolutePath(path) || FileUtil.isWindowsAbsolutePath(path);
    }

    @Deprecated
    public static boolean isUnixAbsolutePath(@NotNull String path) {
        return path.startsWith("/");
    }

    @Deprecated
    public static boolean isWindowsAbsolutePath(@NotNull String path) {
        boolean ok;
        boolean bl = ok = path.length() >= 2 && Character.isLetter(path.charAt(0)) && path.charAt(1) == ':';
        if (ok && path.length() > 2) {
            char separatorChar = path.charAt(2);
            ok = separatorChar == '/' || separatorChar == '\\';
        }
        return ok;
    }

    @Contract(value="null -> null; !null -> !null")
    public static String getLocationRelativeToUserHome(@Nullable String path) {
        return FileUtil.getLocationRelativeToUserHome(path, true);
    }

    @Contract(value="null,_ -> null; !null,_ -> !null")
    public static String getLocationRelativeToUserHome(@Nullable String path, boolean unixOnly) {
        if (path == null) {
            return null;
        }
        if (SystemInfo.isUnix || !unixOnly) {
            File projectDir = new File(path);
            File userHomeDir = new File(SystemProperties.getUserHome());
            if (FileUtil.isAncestor(userHomeDir, projectDir, true)) {
                return '~' + File.separator + FileUtil.getRelativePath(userHomeDir, projectDir);
            }
        }
        return path;
    }

    @NotNull
    public static String expandUserHome(@NotNull String path) {
        if (path.startsWith("~/") || path.startsWith("~\\")) {
            path = SystemProperties.getUserHome() + path.substring(1);
        }
        return path;
    }

    public static File @NotNull [] notNullize(File @Nullable [] files) {
        return FileUtil.notNullize(files, ArrayUtilRt.EMPTY_FILE_ARRAY);
    }

    public static File @NotNull [] notNullize(File @Nullable [] files, File @NotNull [] defaultFiles) {
        return files == null ? defaultFiles : files;
    }

    public static boolean isHashBangLine(@Nullable CharSequence firstCharsIfText, @NotNull String marker) {
        if (firstCharsIfText == null) {
            return false;
        }
        if (!StringUtil.startsWith(firstCharsIfText, "#!")) {
            return false;
        }
        int lineBreak = Strings.indexOf(firstCharsIfText, '\n', 2);
        return lineBreak >= 0 && Strings.indexOf(firstCharsIfText, marker, 2, lineBreak) != -1;
    }

    @NotNull
    public static File createTempDirectory(@NotNull String prefix, @Nullable String suffix) throws IOException {
        return FileUtilRt.createTempDirectory(prefix, suffix);
    }

    @NotNull
    public static File createTempDirectory(@NotNull String prefix, @Nullable String suffix, boolean deleteOnExit) throws IOException {
        return FileUtilRt.createTempDirectory(prefix, suffix, deleteOnExit);
    }

    @NotNull
    public static File createTempDirectory(@NotNull File dir, @NotNull String prefix, @Nullable String suffix) throws IOException {
        return FileUtilRt.createTempDirectory(dir, prefix, suffix);
    }

    @NotNull
    public static File createTempDirectory(@NotNull File dir, @NotNull String prefix, @Nullable String suffix, boolean deleteOnExit) throws IOException {
        return FileUtilRt.createTempDirectory(dir, prefix, suffix, deleteOnExit);
    }

    @NotNull
    public static File createTempFile(@NotNull String prefix, @Nullable String suffix) throws IOException {
        return FileUtilRt.createTempFile(prefix, suffix);
    }

    @NotNull
    public static File createTempFile(@NotNull String prefix, @Nullable String suffix, boolean deleteOnExit) throws IOException {
        return FileUtilRt.createTempFile(prefix, suffix, deleteOnExit);
    }

    @NotNull
    public static File createTempFile(@NotNull File dir, @NotNull String prefix, @Nullable String suffix) throws IOException {
        return FileUtilRt.createTempFile(dir, prefix, suffix);
    }

    @NotNull
    public static File createTempFile(@NotNull File dir, @NotNull String prefix, @Nullable String suffix, boolean create) throws IOException {
        return FileUtilRt.createTempFile(dir, prefix, suffix, create);
    }

    @NotNull
    public static File createTempFile(@NotNull File dir, @NotNull String prefix, @Nullable String suffix, boolean create, boolean deleteOnExit) throws IOException {
        return FileUtilRt.createTempFile(dir, prefix, suffix, create, deleteOnExit);
    }

    @NotNull
    public static String getTempDirectory() {
        return FileUtilRt.getTempDirectory();
    }

    public static void resetCanonicalTempPathCache(@NotNull String tempPath) {
        FileUtilRt.resetCanonicalTempPathCache(tempPath);
    }

    @NotNull
    public static File generateRandomTemporaryPath() throws IOException {
        return FileUtilRt.generateRandomTemporaryPath();
    }

    public static void setExecutable(@NotNull File file) throws IOException {
        Set<PosixFilePermission> permissions;
        PosixFileAttributeView view = Files.getFileAttributeView(file.toPath(), PosixFileAttributeView.class, new LinkOption[0]);
        if (view != null && (permissions = view.readAttributes().permissions()).add(PosixFilePermission.OWNER_EXECUTE)) {
            view.setPermissions(permissions);
        }
    }

    @Deprecated
    @ApiStatus.ScheduledForRemoval(inVersion="2021.1")
    public static void setExecutableAttribute(@NotNull String path, boolean executableFlag) throws IOException {
        FileUtilRt.setExecutableAttribute(path, executableFlag);
    }

    @NotNull
    public static String loadFile(@NotNull File file) throws IOException {
        return FileUtilRt.loadFile(file);
    }

    @NotNull
    public static String loadFile(@NotNull File file, boolean convertLineSeparators) throws IOException {
        return FileUtilRt.loadFile(file, convertLineSeparators);
    }

    @NotNull
    public static String loadFile(@NotNull File file, @Nullable String encoding) throws IOException {
        return FileUtilRt.loadFile(file, encoding);
    }

    @NotNull
    public static String loadFile(@NotNull File file, @NotNull Charset encoding) throws IOException {
        return String.valueOf(FileUtilRt.loadFileText(file, encoding));
    }

    @NotNull
    public static String loadFile(@NotNull File file, @Nullable String encoding, boolean convertLineSeparators) throws IOException {
        return FileUtilRt.loadFile(file, encoding, convertLineSeparators);
    }

    public static char @NotNull [] loadFileText(@NotNull File file) throws IOException {
        return FileUtilRt.loadFileText(file);
    }

    public static char @NotNull [] loadFileText(@NotNull File file, @Nullable String encoding) throws IOException {
        return FileUtilRt.loadFileText(file, encoding);
    }

    public static char @NotNull [] loadText(@NotNull Reader reader, int length) throws IOException {
        return FileUtilRt.loadText(reader, length);
    }

    @NotNull
    public static List<String> loadLines(@NotNull File file) throws IOException {
        return FileUtilRt.loadLines(file);
    }

    @NotNull
    public static List<String> loadLines(@NotNull File file, @Nullable String encoding) throws IOException {
        return FileUtilRt.loadLines(file, encoding);
    }

    @NotNull
    public static List<String> loadLines(@NotNull String path) throws IOException {
        return FileUtilRt.loadLines(path);
    }

    @NotNull
    public static List<String> loadLines(@NotNull String path, @Nullable String encoding) throws IOException {
        return FileUtilRt.loadLines(path, encoding);
    }

    @NotNull
    public static List<String> loadLines(@NotNull BufferedReader reader) throws IOException {
        return FileUtilRt.loadLines(reader);
    }

    public static byte @NotNull [] loadBytes(@NotNull InputStream stream) throws IOException {
        return FileUtilRt.loadBytes(stream);
    }

    public static byte @NotNull [] loadBytes(@NotNull InputStream stream, int length) throws IOException {
        return FileUtilRt.loadBytes(stream, length);
    }

    @NotNull
    public static List<String> splitPath(@NotNull String path) {
        return FileUtil.splitPath(path, File.separatorChar);
    }

    @NotNull
    public static List<String> splitPath(@NotNull String path, char separatorChar) {
        int nextSeparator;
        ArrayList<String> list = new ArrayList<String>();
        int index = 0;
        while ((nextSeparator = path.indexOf(separatorChar, index)) != -1) {
            list.add(path.substring(index, nextSeparator));
            index = nextSeparator + 1;
        }
        list.add(path.substring(index));
        return list;
    }

    public static boolean visitFiles(@NotNull File root, @NotNull Processor<? super File> processor) {
        if (!processor.process(root)) {
            return false;
        }
        File[] children = root.listFiles();
        if (children != null) {
            for (File child : children) {
                if (FileUtil.visitFiles(child, processor)) continue;
                return false;
            }
        }
        return true;
    }

    public static boolean deleteWithRenaming(@NotNull File file) {
        File tempFileNameForDeletion = FileUtil.findSequentNonexistentFile(file.getParentFile(), file.getName(), "");
        boolean success = file.renameTo(tempFileNameForDeletion);
        return FileUtil.delete(success ? tempFileNameForDeletion : file);
    }

    public static boolean isFileSystemCaseSensitive(@NotNull String path) throws FileNotFoundException {
        FileAttributes attributes = FileSystemUtil.getAttributes(path);
        if (attributes == null) {
            throw new FileNotFoundException(path);
        }
        FileAttributes upper = FileSystemUtil.getAttributes(Strings.toUpperCase(path));
        FileAttributes lower = FileSystemUtil.getAttributes(Strings.toLowerCase(path));
        return !attributes.equals(upper) || !attributes.equals(lower);
    }

    @NotNull
    public static String getUrl(@NotNull File file) {
        try {
            return file.toURI().toURL().toExternalForm();
        }
        catch (MalformedURLException e) {
            return "file://" + file.getAbsolutePath();
        }
    }

    @NotNull
    public static URI fileToUri(@NotNull File file) {
        return FileUtilRt.fileToUri(file);
    }

    private static class Lazy {
        private static final JBTreeTraverser<File> FILE_TRAVERSER = JBTreeTraverser.from(file -> file == null ? Collections.emptySet() : JBIterable.of(file.listFiles()));

        private Lazy() {
        }
    }
}

