/*
 * Decompiled with CFR 0.152.
 */
package android.net.wifi.aware;

import java.nio.BufferOverflowException;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;

public class TlvBufferUtils {
    private TlvBufferUtils() {
    }

    public static boolean isValid(byte[] array2, int typeSize, int lengthSize) {
        return TlvBufferUtils.isValidEndian(array2, typeSize, lengthSize, ByteOrder.BIG_ENDIAN);
    }

    public static boolean isValidEndian(byte[] array2, int typeSize, int lengthSize, ByteOrder byteOrder) {
        if (typeSize < 0 || typeSize > 2) {
            throw new IllegalArgumentException("Invalid arguments - typeSize must be 0, 1, or 2: typeSize=" + typeSize);
        }
        if (lengthSize <= 0 || lengthSize > 2) {
            throw new IllegalArgumentException("Invalid arguments - lengthSize must be 1 or 2: lengthSize=" + lengthSize);
        }
        if (array2 == null) {
            return true;
        }
        int nextTlvIndex = 0;
        while (nextTlvIndex + typeSize + lengthSize <= array2.length) {
            nextTlvIndex += typeSize;
            if (lengthSize == 1) {
                nextTlvIndex += lengthSize + array2[nextTlvIndex];
                continue;
            }
            nextTlvIndex += lengthSize + TlvBufferUtils.peekShort(array2, nextTlvIndex, byteOrder);
        }
        return nextTlvIndex == array2.length;
    }

    private static void pokeShort(byte[] dst, int offset, short value, ByteOrder order) {
        if (order == ByteOrder.BIG_ENDIAN) {
            dst[offset++] = (byte)(value >> 8 & 0xFF);
            dst[offset] = (byte)(value >> 0 & 0xFF);
        } else {
            dst[offset++] = (byte)(value >> 0 & 0xFF);
            dst[offset] = (byte)(value >> 8 & 0xFF);
        }
    }

    private static void pokeInt(byte[] dst, int offset, int value, ByteOrder order) {
        if (order == ByteOrder.BIG_ENDIAN) {
            dst[offset++] = (byte)(value >> 24 & 0xFF);
            dst[offset++] = (byte)(value >> 16 & 0xFF);
            dst[offset++] = (byte)(value >> 8 & 0xFF);
            dst[offset] = (byte)(value >> 0 & 0xFF);
        } else {
            dst[offset++] = (byte)(value >> 0 & 0xFF);
            dst[offset++] = (byte)(value >> 8 & 0xFF);
            dst[offset++] = (byte)(value >> 16 & 0xFF);
            dst[offset] = (byte)(value >> 24 & 0xFF);
        }
    }

    private static short peekShort(byte[] src, int offset, ByteOrder order) {
        if (order == ByteOrder.BIG_ENDIAN) {
            return (short)(src[offset] << 8 | src[offset + 1] & 0xFF);
        }
        return (short)(src[offset + 1] << 8 | src[offset] & 0xFF);
    }

    private static int peekInt(byte[] src, int offset, ByteOrder order) {
        if (order == ByteOrder.BIG_ENDIAN) {
            return (src[offset++] & 0xFF) << 24 | (src[offset++] & 0xFF) << 16 | (src[offset++] & 0xFF) << 8 | (src[offset] & 0xFF) << 0;
        }
        return (src[offset++] & 0xFF) << 0 | (src[offset++] & 0xFF) << 8 | (src[offset++] & 0xFF) << 16 | (src[offset] & 0xFF) << 24;
    }

    public static class TlvIterable
    implements Iterable<TlvElement> {
        private int mTypeSize;
        private int mLengthSize;
        private ByteOrder mByteOrder = ByteOrder.BIG_ENDIAN;
        private byte[] mArray;
        private int mArrayLength;

        public TlvIterable(int typeSize, int lengthSize, byte[] array2) {
            if (typeSize < 0 || typeSize > 2 || lengthSize <= 0 || lengthSize > 2) {
                throw new IllegalArgumentException("Invalid sizes - typeSize=" + typeSize + ", lengthSize=" + lengthSize);
            }
            this.mTypeSize = typeSize;
            this.mLengthSize = lengthSize;
            this.mArray = array2;
            this.mArrayLength = array2 == null ? 0 : array2.length;
        }

        public void setByteOrder(ByteOrder byteOrder) {
            this.mByteOrder = byteOrder;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("[");
            boolean first = true;
            for (TlvElement tlv : this) {
                if (!first) {
                    builder.append(",");
                }
                first = false;
                builder.append(" (");
                if (this.mTypeSize != 0) {
                    builder.append("T=" + tlv.type + ",");
                }
                builder.append("L=" + tlv.length + ") ");
                if (tlv.length == 0) {
                    builder.append("<null>");
                } else if (tlv.length == 1) {
                    builder.append(tlv.getByte());
                } else if (tlv.length == 2) {
                    builder.append(tlv.getShort());
                } else if (tlv.length == 4) {
                    builder.append(tlv.getInt());
                } else {
                    builder.append("<bytes>");
                }
                if (tlv.length == 0) continue;
                builder.append(" (S='" + tlv.getString() + "')");
            }
            builder.append("]");
            return builder.toString();
        }

        public List<byte[]> toList() {
            ArrayList<byte[]> list = new ArrayList<byte[]>();
            for (TlvElement tlv : this) {
                list.add(Arrays.copyOfRange(tlv.mRefArray, tlv.offset, tlv.offset + tlv.length));
            }
            return list;
        }

        @Override
        public Iterator<TlvElement> iterator() {
            return new Iterator<TlvElement>(){
                private int mOffset = 0;

                @Override
                public boolean hasNext() {
                    return this.mOffset < mArrayLength;
                }

                @Override
                public TlvElement next() {
                    if (!this.hasNext()) {
                        throw new NoSuchElementException();
                    }
                    short type = 0;
                    if (mTypeSize == 1) {
                        type = mArray[this.mOffset];
                    } else if (mTypeSize == 2) {
                        type = TlvBufferUtils.peekShort(mArray, this.mOffset, mByteOrder);
                    }
                    this.mOffset += mTypeSize;
                    short length = 0;
                    if (mLengthSize == 1) {
                        length = mArray[this.mOffset];
                    } else if (mLengthSize == 2) {
                        length = TlvBufferUtils.peekShort(mArray, this.mOffset, mByteOrder);
                    }
                    this.mOffset += mLengthSize;
                    TlvElement tlv = new TlvElement(type, length, mArray, this.mOffset);
                    tlv.byteOrder = mByteOrder;
                    this.mOffset += length;
                    return tlv;
                }

                @Override
                public void remove() {
                    throw new UnsupportedOperationException();
                }
            };
        }
    }

    public static class TlvElement {
        public int type;
        public int length;
        public ByteOrder byteOrder = ByteOrder.BIG_ENDIAN;
        private byte[] mRefArray;
        public int offset;

        private TlvElement(int type, int length, byte[] refArray, int offset) {
            this.type = type;
            this.length = length;
            this.mRefArray = refArray;
            this.offset = offset;
            if (offset + length > refArray.length) {
                throw new BufferOverflowException();
            }
        }

        public byte[] getRawData() {
            return Arrays.copyOfRange(this.mRefArray, this.offset, this.offset + this.length);
        }

        public byte getByte() {
            if (this.length != 1) {
                throw new IllegalArgumentException("Accesing a byte from a TLV element of length " + this.length);
            }
            return this.mRefArray[this.offset];
        }

        public short getShort() {
            if (this.length != 2) {
                throw new IllegalArgumentException("Accesing a short from a TLV element of length " + this.length);
            }
            return TlvBufferUtils.peekShort(this.mRefArray, this.offset, this.byteOrder);
        }

        public int getInt() {
            if (this.length != 4) {
                throw new IllegalArgumentException("Accesing an int from a TLV element of length " + this.length);
            }
            return TlvBufferUtils.peekInt(this.mRefArray, this.offset, this.byteOrder);
        }

        public String getString() {
            return new String(this.mRefArray, this.offset, this.length);
        }
    }

    public static class TlvConstructor {
        private int mTypeSize;
        private int mLengthSize;
        private ByteOrder mByteOrder = ByteOrder.BIG_ENDIAN;
        private byte[] mArray;
        private int mArrayLength;
        private int mPosition;

        public TlvConstructor(int typeSize, int lengthSize) {
            if (typeSize < 0 || typeSize > 2 || lengthSize <= 0 || lengthSize > 2) {
                throw new IllegalArgumentException("Invalid sizes - typeSize=" + typeSize + ", lengthSize=" + lengthSize);
            }
            this.mTypeSize = typeSize;
            this.mLengthSize = lengthSize;
            this.mPosition = 0;
        }

        public TlvConstructor setByteOrder(ByteOrder byteOrder) {
            this.mByteOrder = byteOrder;
            return this;
        }

        public TlvConstructor wrap(byte[] array2) {
            this.mArray = array2;
            this.mArrayLength = array2 == null ? 0 : array2.length;
            this.mPosition = 0;
            return this;
        }

        public TlvConstructor allocate(int capacity) {
            this.mArray = new byte[capacity];
            this.mArrayLength = capacity;
            this.mPosition = 0;
            return this;
        }

        public TlvConstructor allocateAndPut(List<byte[]> list) {
            if (list != null) {
                int size = 0;
                for (byte[] field : list) {
                    size += this.mTypeSize + this.mLengthSize;
                    if (field == null) continue;
                    size += field.length;
                }
                this.allocate(size);
                for (byte[] field : list) {
                    this.putByteArray(0, field);
                }
            }
            return this;
        }

        public TlvConstructor putByte(int type, byte b) {
            this.checkLength(1);
            this.addHeader(type, 1);
            this.mArray[this.mPosition++] = b;
            return this;
        }

        public TlvConstructor putRawByte(byte b) {
            this.checkRawLength(1);
            this.mArray[this.mPosition++] = b;
            return this;
        }

        public TlvConstructor putByteArray(int type, byte[] array2, int offset, int length) {
            this.checkLength(length);
            this.addHeader(type, length);
            if (length != 0) {
                System.arraycopy(array2, offset, this.mArray, this.mPosition, length);
            }
            this.mPosition += length;
            return this;
        }

        public TlvConstructor putByteArray(int type, byte[] array2) {
            return this.putByteArray(type, array2, 0, array2 == null ? 0 : array2.length);
        }

        public TlvConstructor putRawByteArray(byte[] array2) {
            if (array2 == null) {
                return this;
            }
            this.checkRawLength(array2.length);
            System.arraycopy(array2, 0, this.mArray, this.mPosition, array2.length);
            this.mPosition += array2.length;
            return this;
        }

        public TlvConstructor putZeroLengthElement(int type) {
            this.checkLength(0);
            this.addHeader(type, 0);
            return this;
        }

        public TlvConstructor putShort(int type, short data) {
            this.checkLength(2);
            this.addHeader(type, 2);
            TlvBufferUtils.pokeShort(this.mArray, this.mPosition, data, this.mByteOrder);
            this.mPosition += 2;
            return this;
        }

        public TlvConstructor putInt(int type, int data) {
            this.checkLength(4);
            this.addHeader(type, 4);
            TlvBufferUtils.pokeInt(this.mArray, this.mPosition, data, this.mByteOrder);
            this.mPosition += 4;
            return this;
        }

        public TlvConstructor putString(int type, String data) {
            byte[] bytes = null;
            int length = 0;
            if (data != null) {
                bytes = data.getBytes();
                length = bytes.length;
            }
            return this.putByteArray(type, bytes, 0, length);
        }

        public byte[] getArray() {
            return Arrays.copyOf(this.mArray, this.getActualLength());
        }

        private int getActualLength() {
            return this.mPosition;
        }

        private void checkLength(int dataLength) {
            if (this.mPosition + this.mTypeSize + this.mLengthSize + dataLength > this.mArrayLength) {
                throw new BufferOverflowException();
            }
        }

        private void checkRawLength(int dataLength) {
            if (this.mPosition + dataLength > this.mArrayLength) {
                throw new BufferOverflowException();
            }
        }

        private void addHeader(int type, int length) {
            if (this.mTypeSize == 1) {
                this.mArray[this.mPosition] = (byte)type;
            } else if (this.mTypeSize == 2) {
                TlvBufferUtils.pokeShort(this.mArray, this.mPosition, (short)type, this.mByteOrder);
            }
            this.mPosition += this.mTypeSize;
            if (this.mLengthSize == 1) {
                this.mArray[this.mPosition] = (byte)length;
            } else if (this.mLengthSize == 2) {
                TlvBufferUtils.pokeShort(this.mArray, this.mPosition, (short)length, this.mByteOrder);
            }
            this.mPosition += this.mLengthSize;
        }
    }
}

