/*
 * Decompiled with CFR 0.152.
 */
package com.intellij.openapi.vcs.impl;

import com.intellij.openapi.Disposable;
import com.intellij.openapi.application.Application;
import com.intellij.openapi.application.ApplicationManager;
import com.intellij.openapi.application.ModalityState;
import com.intellij.openapi.diagnostic.Logger;
import com.intellij.openapi.editor.Document;
import com.intellij.openapi.editor.colors.ColorKey;
import com.intellij.openapi.editor.colors.EditorColorsListener;
import com.intellij.openapi.editor.colors.EditorColorsManager;
import com.intellij.openapi.editor.colors.EditorColorsScheme;
import com.intellij.openapi.editor.event.DocumentEvent;
import com.intellij.openapi.editor.event.DocumentListener;
import com.intellij.openapi.extensions.AreaInstance;
import com.intellij.openapi.fileEditor.FileDocumentManager;
import com.intellij.openapi.fileEditor.FileDocumentManagerListener;
import com.intellij.openapi.project.Project;
import com.intellij.openapi.project.ProjectManager;
import com.intellij.openapi.startup.StartupManager;
import com.intellij.openapi.util.Disposer;
import com.intellij.openapi.util.Key;
import com.intellij.openapi.vcs.FileStatus;
import com.intellij.openapi.vcs.FileStatusListener;
import com.intellij.openapi.vcs.FileStatusManager;
import com.intellij.openapi.vcs.ProjectLevelVcsManager;
import com.intellij.openapi.vcs.impl.FileStatusProvider;
import com.intellij.openapi.vcs.impl.VcsFileStatusProvider;
import com.intellij.openapi.vfs.NonPhysicalFileSystem;
import com.intellij.openapi.vfs.VFileProperty;
import com.intellij.openapi.vfs.VirtualFile;
import com.intellij.util.ThreeState;
import com.intellij.util.containers.ContainerUtil;
import com.intellij.util.messages.MessageBusConnection;
import java.awt.Color;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import org.jetbrains.annotations.NotNull;

public final class FileStatusManagerImpl
extends FileStatusManager
implements Disposable {
    private static final Logger LOG = Logger.getInstance(FileStatusManagerImpl.class);
    private final Map<VirtualFile, FileStatus> myCachedStatuses = Collections.synchronizedMap(new HashMap());
    private final Map<VirtualFile, Boolean> myWhetherExactlyParentToChanged = Collections.synchronizedMap(new HashMap());
    private final Project myProject;
    private final List<FileStatusListener> myListeners = ContainerUtil.createLockFreeCopyOnWriteList();
    private final FileStatusProvider myFileStatusProvider;

    public FileStatusManagerImpl(@NotNull Project project) {
        this.myProject = project;
        this.myFileStatusProvider = VcsFileStatusProvider.getInstance(project);
        MessageBusConnection projectBus = project.getMessageBus().connect();
        projectBus.subscribe(EditorColorsManager.TOPIC, (Object)new EditorColorsListener(){

            public void globalSchemeChange(EditorColorsScheme scheme) {
                FileStatusManagerImpl.this.fileStatusesChanged();
            }
        });
        projectBus.subscribe(ProjectLevelVcsManager.VCS_CONFIGURATION_CHANGED, this::fileStatusesChanged);
        if (!project.isDefault()) {
            StartupManager.getInstance((Project)project).runAfterOpened(this::fileStatusesChanged);
        }
        FileStatusProvider.EP_NAME.addChangeListener((AreaInstance)this.myProject, this::fileStatusesChanged, (Disposable)project);
    }

    public FileStatus calcStatus(@NotNull VirtualFile virtualFile) {
        for (FileStatusProvider extension : FileStatusProvider.EP_NAME.getExtensions((AreaInstance)this.myProject)) {
            FileStatus status = extension.getFileStatus(virtualFile);
            if (status == null) continue;
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("File status for file [%s] from provider %s: %s", virtualFile, extension.getClass().getName(), status));
            }
            return status;
        }
        if (virtualFile.isInLocalFileSystem()) {
            FileStatus status = this.myFileStatusProvider.getFileStatus(virtualFile);
            if (LOG.isDebugEnabled()) {
                LOG.debug(String.format("File status for file [%s] from default provider %s: %s", virtualFile, this.myFileStatusProvider, status));
            }
            return status;
        }
        FileStatus defaultStatus = FileStatusManagerImpl.getDefaultStatus(virtualFile);
        if (LOG.isDebugEnabled()) {
            LOG.debug(String.format("Default status for file [%s]: %s", virtualFile, defaultStatus));
        }
        return defaultStatus;
    }

    @NotNull
    public static FileStatus getDefaultStatus(@NotNull VirtualFile file) {
        return file.isValid() && file.is(VFileProperty.SPECIAL) ? FileStatus.IGNORED : FileStatus.NOT_CHANGED;
    }

    public void dispose() {
        this.myCachedStatuses.clear();
    }

    public void addFileStatusListener(@NotNull FileStatusListener listener2) {
        this.myListeners.add(listener2);
    }

    public void addFileStatusListener(final @NotNull FileStatusListener listener2, @NotNull Disposable parentDisposable) {
        this.addFileStatusListener(listener2);
        Disposer.register((Disposable)parentDisposable, (Disposable)new Disposable(){

            public void dispose() {
                FileStatusManagerImpl.this.removeFileStatusListener(listener2);
            }
        });
    }

    public void fileStatusesChanged() {
        if (this.myProject.isDisposed()) {
            return;
        }
        if (!ApplicationManager.getApplication().isDispatchThread()) {
            ApplicationManager.getApplication().invokeLater(() -> this.fileStatusesChanged(), ModalityState.any());
            return;
        }
        this.myCachedStatuses.clear();
        this.myWhetherExactlyParentToChanged.clear();
        for (FileStatusListener listener2 : this.myListeners) {
            listener2.fileStatusesChanged();
        }
    }

    private void cacheChangedFileStatus(VirtualFile virtualFile, FileStatus fs) {
        this.myCachedStatuses.put(virtualFile, fs);
        if (FileStatus.NOT_CHANGED.equals(fs)) {
            ThreeState parentingStatus = this.myFileStatusProvider.getNotChangedDirectoryParentingStatus(virtualFile);
            if (ThreeState.YES.equals((Object)parentingStatus)) {
                this.myWhetherExactlyParentToChanged.put(virtualFile, true);
            } else if (ThreeState.UNSURE.equals((Object)parentingStatus)) {
                this.myWhetherExactlyParentToChanged.put(virtualFile, false);
            }
        } else {
            this.myWhetherExactlyParentToChanged.remove(virtualFile);
        }
    }

    public void fileStatusChanged(VirtualFile file) {
        Application application = ApplicationManager.getApplication();
        if (!application.isDispatchThread() && !application.isUnitTestMode()) {
            ApplicationManager.getApplication().invokeLater(() -> this.fileStatusChanged(file));
            return;
        }
        if (file == null || !file.isValid()) {
            return;
        }
        FileStatus cachedStatus = this.getCachedStatus(file);
        if (cachedStatus == FileStatusNull.INSTANCE) {
            return;
        }
        if (cachedStatus == null) {
            this.cacheChangedFileStatus(file, FileStatusNull.INSTANCE);
            return;
        }
        FileStatus newStatus = this.calcStatus(file);
        if (cachedStatus == newStatus) {
            return;
        }
        this.cacheChangedFileStatus(file, newStatus);
        for (FileStatusListener listener2 : this.myListeners) {
            listener2.fileStatusChanged(file);
        }
    }

    public FileStatus getStatus(@NotNull VirtualFile file) {
        if (file.getFileSystem() instanceof NonPhysicalFileSystem) {
            return FileStatus.SUPPRESSED;
        }
        FileStatus status = this.getCachedStatus(file);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Cached status for file [" + file + "] is " + status);
        }
        if (status == null || status == FileStatusNull.INSTANCE) {
            status = this.calcStatus(file);
            this.cacheChangedFileStatus(file, status);
        }
        return status;
    }

    public FileStatus getCachedStatus(VirtualFile file) {
        return this.myCachedStatuses.get(file);
    }

    public void removeFileStatusListener(@NotNull FileStatusListener listener2) {
        this.myListeners.remove(listener2);
    }

    public Color getNotChangedDirectoryColor(@NotNull VirtualFile file) {
        return this.getRecursiveStatus(file).getColor();
    }

    @NotNull
    public FileStatus getRecursiveStatus(@NotNull VirtualFile file) {
        FileStatus status = super.getRecursiveStatus(file);
        if (status != FileStatus.NOT_CHANGED || !file.isValid() || !file.isDirectory()) {
            return status;
        }
        Boolean immediate = this.myWhetherExactlyParentToChanged.get(file);
        if (immediate == null) {
            return status;
        }
        return immediate != false ? FileStatus.NOT_CHANGED_IMMEDIATE : FileStatus.NOT_CHANGED_RECURSIVE;
    }

    public void refreshFileStatusFromDocument(VirtualFile file, Document doc) {
        this.myFileStatusProvider.refreshFileStatusFromDocument(file, doc);
    }

    static final class FileStatusManagerDocumentListener
    implements FileDocumentManagerListener,
    DocumentListener {
        private final Key<Boolean> CHANGED = Key.create((String)"FileStatusManagerDocumentListener.document.changed");

        FileStatusManagerDocumentListener() {
        }

        public void documentChanged(@NotNull DocumentEvent event) {
            Document document = event.getDocument();
            if (document.isInBulkUpdate()) {
                document.putUserData(this.CHANGED, (Object)Boolean.TRUE);
            } else {
                FileStatusManagerDocumentListener.refreshFileStatus(document);
            }
        }

        public void bulkUpdateFinished(@NotNull Document document) {
            if (document.getUserData(this.CHANGED) != null) {
                document.putUserData(this.CHANGED, null);
                FileStatusManagerDocumentListener.refreshFileStatus(document);
            }
        }

        public void unsavedDocumentDropped(@NotNull Document document) {
            FileStatusManagerDocumentListener.refreshFileStatus(document);
        }

        private static void refreshFileStatus(@NotNull Document document) {
            VirtualFile file = FileDocumentManager.getInstance().getFile(document);
            if (file == null) {
                return;
            }
            ProjectManager projectManager = ProjectManager.getInstanceIfCreated();
            if (projectManager == null) {
                return;
            }
            for (Project project : projectManager.getOpenProjects()) {
                FileStatusManager.getInstance((Project)project).refreshFileStatusFromDocument(file, document);
            }
        }
    }

    private static class FileStatusNull
    implements FileStatus {
        private static final FileStatus INSTANCE = new FileStatusNull();

        private FileStatusNull() {
        }

        public String getText() {
            throw new AssertionError((Object)"Should not be called");
        }

        public Color getColor() {
            throw new AssertionError((Object)"Should not be called");
        }

        @NotNull
        public ColorKey getColorKey() {
            throw new AssertionError((Object)"Should not be called");
        }

        @NotNull
        public String getId() {
            throw new AssertionError((Object)"Should not be called");
        }
    }
}

